import copy
import io
import logging
import time
import threading

import pandas
from django.db import transaction
from django.utils import timezone
from rest_framework.mixins import ListModelMixin
from rest_framework.parsers import FileUploadParser
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.admin.permissions import AdminPermissionFactory, AdminPermissionCode
from cars.admin.serializers.car_document import (
    CarDocumentAssignmentsListViewArgumentsSerializer,
    CarDocumentAssignmentSerializer,
    CarDocumentSerializer,
    CarDocumentsListViewArgumentsSerializer,
)
from cars.carsharing.core.documents_manager import DocumentFilesManager
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.insurance_manager import CarInsuranceManager
from cars.carsharing.core.registry_manager import CarRegistryManager
from cars.carsharing.models.car_document import CarDocument, CarDocumentAssignment
from cars.django.pagination import Pagination
from cars.service_app.core.assembly_manager import AssemblyManager
from .base import AdminAPIView


LOGGER = logging.getLogger(__name__)


class CarDocumentSimView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_docs')]

    parser_classes = [FileUploadParser]

    def do_put(self, request, *args, **kwargs):  # pylint:disable=unused-argument
        try:
            excel_fileobj = io.BytesIO(request.FILES['file'].read())
            df = pandas.read_excel(excel_fileobj)
        except Exception:
            return Response(
                {
                    'status': 'error',
                    'error': 'excel.corrupt_file',
                },
            )

        addition_payload = []

        for index, row in df.iterrows():
            icc = row.get('ICC')
            phone_number = row.get('Номер телефона')
            if not isinstance(phone_number, str):
                phone_number = str(phone_number)

            if not icc or not phone_number:
                continue

            if len(phone_number) == 10:
                phone_number = '+7{}'.format(phone_number)
            elif len(phone_number) == 11:
                phone_number = '+{}'.format(phone_number)

            addition_payload.append(
                {
                    'icc': icc,
                    'phone_number': phone_number,
                }
            )

        manager = AssemblyManager(request.user)

        return Response(
            {
                'status': 'success',
                'num_added': manager.bulk_add_sim_cards(addition_payload),
            },
        )


class CarInsuranceUploadView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_docs')]

    parser_classes = [FileUploadParser]

    def do_put(self, request, *args, **kwargs):  # pylint:disable=unused-argument
        manager = CarInsuranceManager(request.user)
        try:
            t = threading.Thread(target=self._start_uploads, args=(request, manager))
            t.start()
        except Exception:
            LOGGER.exception('unable to parse')
            return Response(
                {
                    'status': 'error',
                    'error': 'excel.corrupt_file',
                },
            )

        return Response(
            {
                'status': 'success',
            }
        )

    def _start_uploads(self, request, manager):
        excel_fileobj = io.BytesIO(request.FILES['file'].read())
        try:
            df = pandas.read_excel(copy.deepcopy(excel_fileobj), sheet_name=None)
            for page_name, page_content in df.items():
                manager.bulk_add_insurance_from_df(page_content)
        except Exception:
            LOGGER.error('unable to get the sheet')


class CreateCarsByVINNumbersView(AdminAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.ACCESS)]
    parser_classes = [FileUploadParser]

    def do_put(self, request, *args, **kwargs):  # pylint:disable=unused-argument
        try:
            excel_fileobj = io.BytesIO(request.FILES['file'].read())
            df = pandas.read_excel(excel_fileobj)
        except Exception:
            return Response(
                {
                    'status': 'error',
                    'error': 'excel.corrupt_file',
                },
            )

        manager = CarRegistryManager(request.user)
        num_added = manager.bulk_add_and_update_cars_from_df(df)

        return Response(
            {
                'status': 'success',
                'num_added': num_added,
            }
        )


class CarDocumentsListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_docs')]

    arguments_serializer_class = CarDocumentsListViewArgumentsSerializer
    pagination_class = Pagination
    serializer_class = CarDocumentSerializer

    def get_queryset(self):
        qs = (
            CarDocument.objects
            .with_related()
            .select_related('added_by__registration_state')
            .order_by('-added_at')
        )

        filter_params = {}
        if 'type' in self.request.arguments:
            filter_params['type'] = self.request.arguments['type']

        if filter_params:
            qs = qs.filter(**filter_params)

        return qs

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class CarDocumentDetachView(AdminAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.ACCESS)]

    def do_post(self, request, car_id, document_id):  # pylint:disable=unused-argument
        obj = (
            CarDocumentAssignment.objects
            .filter(
                car_id=car_id,
                document_id=document_id,
                unassigned_at__isnull=True,
            )
            .first()
        )

        if obj is None:
            return Response(
                status=HTTP_400_BAD_REQUEST,
            )

        upd = CarUpdater(obj.car)
        upd.unassign_document(obj, request.user)

        return Response()


class CarDocumentAssignmentsListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_docs')]

    arguments_serializer_class = CarDocumentAssignmentsListViewArgumentsSerializer
    pagination_class = Pagination
    serializer_class = CarDocumentAssignmentSerializer

    def get_queryset(self):
        qs = (
            CarDocumentAssignment.objects
            .with_related()
            .select_related(
                'car__telematics_state',
                'car__location',
                'car__model',
                'document',
                'document__added_by__registration_state',
                'assigned_by__registration_state',
                'unassigned_by__registration_state',
            )
            .order_by('-assigned_at')
        )

        filter_params = {}
        if 'type' in self.request.arguments:
            filter_params['document__type'] = self.request.arguments['type']
        if 'car_id' in self.request.arguments:
            filter_params['car_id'] = self.request.arguments['car_id']
        if 'assigned_by' in self.request.arguments:
            filter_params['assigned_by'] = self.request.arguments['assigned_by']
        if 'unassigned_by' in self.request.arguments:
            filter_params['unassigned_by'] = self.request.arguments['unassigned_by']

        if filter_params:
            qs = qs.filter(**filter_params)

        return qs

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class CarDocumentFileUploadView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_docs')]

    def do_post(self, request):
        try:
            filename = request.data['filename']
            data = request.data['data']
            type_ = DocumentFilesManager.DocumentFileType(request.data['type'])
        except (KeyError, ValueError):
            return Response(status=HTTP_400_BAD_REQUEST)

        manager = DocumentFilesManager.from_settings()
        manager.save_document(
            filename=filename,
            data=data,
            type_=type_,
        )
        return Response()
