from django.shortcuts import get_object_or_404
from rest_framework.response import Response
from rest_framework.status import HTTP_404_NOT_FOUND

import cars.settings
from cars.carsharing.models import Car
from cars.django.util import ReqAnsLogger
from cars.orders.models import OrderItem
from ..serializers.order import (
    CarAllInfoByIMEIArgumentsSerializer,
    CarAllInfoByIMEIResponseSerializer
)
from .base import AdminAPIView


class CarInfoByIMEIByTimeView(AdminAPIView):

    arguments_serializer_class = CarAllInfoByIMEIArgumentsSerializer
    serializer_class = CarAllInfoByIMEIResponseSerializer

    permission_classes = []

    reqans_logger_policy = ReqAnsLogger.Policy.NONE

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        timestamp, imei = request.arguments['timestamp'], request.arguments['imei']

        # last of the order items for this car which've started before timestamp
        try:
            last_reservation_item = (
                OrderItem.objects
                .using(cars.settings.DB_RO_ID)
                .filter(
                    started_at__lte=timestamp,
                    carsharing_reservation__car__imei=imei,
                )
                .select_related(
                    'order',
                )
                .latest('started_at')
            )

            items = last_reservation_item.order.items

            last_order_item = (
                items
                .filter(
                    started_at__lte=timestamp,
                )
                .latest('started_at')
            )
        except OrderItem.DoesNotExist:
            last_order_item = None

        last_order_item_finished_before = (
            (last_order_item is not None) and
            (last_order_item.finished_at is not None) and
            (last_order_item.finished_at < timestamp)
        )
        if (last_order_item is None) or last_order_item_finished_before:
            try:
                next_reservation_item = (
                    OrderItem.objects
                    .using(cars.settings.DB_RO_ID)
                    .filter(
                        started_at__gt=timestamp,
                        carsharing_reservation__car__imei=imei,
                    )
                    .earliest('started_at')
                )

                items = next_reservation_item.order.items

                next_order_item = (
                    items
                    .filter(
                        started_at__gt=timestamp,
                    )
                    .earliest('started_at')
                )
            except OrderItem.DoesNotExist:
                next_order_item = None

            if last_order_item:
                timestamp_start = last_order_item.finished_at
            else:
                timestamp_start = None

            if next_order_item is not None:
                timestamp_finish = next_order_item.started_at
            else:
                timestamp_finish = None

            car = get_object_or_404(Car, imei=imei)

            return Response(
                {
                    'action_id': None,
                    'session_id': None,
                    'car_id': str(car.id),
                    'car_status': Car.Status.AVAILABLE.value,
                    'timestamp_start': (
                        timestamp_start.timestamp() if timestamp_start else None
                    ),
                    'timestamp_finish': (
                        timestamp_finish.timestamp() if timestamp_finish else None
                    ),
                }
            )

        return Response(self.serializer_class(last_order_item).data)

    def _check_acl(self, user):
        return True


class CarStatusByIMEIView(AdminAPIView):

    permission_classes = []

    reqans_logger_policy = ReqAnsLogger.Policy.NONE

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        imei = request.GET.get('imei')
        car = (
            Car.objects
            .using(cars.settings.DB_RO_ID)
            .filter(imei=imei)
            .first()
        )
        if not car:
            return Response(status=HTTP_404_NOT_FOUND)

        return Response(
            {
                'id': str(car.id),
                'status': car.status,
            }
        )

    def _check_acl(self, user):
        return True


class CarRideByIMEIView(AdminAPIView):

    permission_classes = []

    reqans_logger_policy = ReqAnsLogger.Policy.NONE

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        imei = request.GET.get('imei')

        car_id = None
        current_ride_id = None
        current_user_id = None
        current_order_id = None
        try:
            current_ride_order_item = (
                OrderItem.objects
                .using(cars.settings.DB_RO_ID)
                .filter(
                    type=OrderItem.Type.CARSHARING_RIDE.value,
                    carsharing_ride__car__imei=imei,
                )
                .select_related(
                    'order',
                    'carsharing_ride',
                )
                .latest('started_at')
            )
        except OrderItem.DoesNotExist:
            current_ride_order_item = None

        if current_ride_order_item:
            car_id = str(current_ride_order_item.carsharing_ride.car_id)
            current_ride_id = str(current_ride_order_item.id)
            current_user_id = str(current_ride_order_item.order.user_id)
            current_order_id = str(current_ride_order_item.order.id)
            if current_ride_order_item.finished_at is not None:
                current_ride_id = 'post-' + str(current_ride_id)

        return Response(
            {
                'id': car_id,
                'current_order_id': current_order_id,
                'current_ride_id': current_ride_id,
                'current_user_id': current_user_id,
            },
        )

    def _check_acl(self, user):
        return True
