from django.utils import timezone
from django.db.models import Q
from rest_framework.exceptions import NotFound
from rest_framework.mixins import ListModelMixin, RetrieveModelMixin
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.core.tariff_manager import CarsharingTariffPicker
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models import CarsharingTagPhoto
from cars.carsharing.models.acceptance import CarsharingAcceptancePhoto
from cars.carsharing.models.car import Car
from cars.carsharing.models.car_assembly import CarAssemblyPhoto
from cars.carsharing.models.car_model import CarModel
from cars.core.util import import_class
from cars.django.pagination import Pagination
from cars.django.util import ReqAnsLogger
from cars.eka.core.processor import FuelCardProcessor
from cars.eka.models import FuelCardActivation
from cars.orders.core.order_payment_processor import OrderPaymentProcessor
from ..permissions import AdminPermissionCode, AdminPermissionFactory
from ..serializers.car_assembly import CarAssemblyPhotoSerializer, CarAssemblyStageSerializer
from ..serializers.car_tag_photo import CarsharingTagPhotoSerializer
from ..serializers.carsharing import (
    CarListViewArgumentsSerializer,
    CarModelSerializer,
    CarSerializer,
    UpdateFuelCardArgumentsSerializer,
)
from ..serializers.carsharing_simple import SimpleCarSerializer
from ..serializers.order import CarsharingAcceptancePhotoSerializer
from .base import AdminAPIView, AdminAndCallcenterAPIView
from cars.orders.models import Order


class CarListView(ListModelMixin, AdminAndCallcenterAPIView):

    arguments_serializer_class = CarListViewArgumentsSerializer
    serializer_class = SimpleCarSerializer

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.VIEW_ALL_CARS)]
    reqans_logger_policy = ReqAnsLogger.Policy.NONE

    def get_queryset(self):
        qs = (
            Car.objects
            .select_related('location', 'model', 'telematics_state')
        )

        query = self.request.arguments.get('q')
        if query:
            lookup_fields = [
                'number'
            ]
            query_normalized = query.lower()

            dbquery = Q()
            for lookup_field in lookup_fields:
                dbquery |= Q(**{
                    '{}__icontains'.format(lookup_field): query_normalized,
                })

            qs = qs.filter(dbquery)

        return qs

    def do_get(self, _):
        cars = self.get_queryset()
        data = {
            'cars': SimpleCarSerializer(cars, many=True).data,
        }
        return Response(data)


class CarAcceptancePhotosView(ListModelMixin, AdminAPIView):

    pagination_class = Pagination
    serializer_class = CarsharingAcceptancePhotoSerializer

    def get_queryset(self):
        return (
            CarsharingAcceptancePhoto.objects
            .filter(
                acceptance__car_id=self.kwargs['car_id'],
            )
            .select_related('acceptance__order_item')
            .order_by('-submitted_at')
        )

    def do_get(self, request, car_id):
        return self.list(request, car_id)


class CarAssemblyPhotosView(AdminAPIView):

    def get_queryset(self):
        return Car.objects

    def do_get(self, request, car_id):  # pylint:disable=unused-argument
        car = (
            Car.objects
            .filter(id=car_id)
            .prefetch_related('model__assembly_stages')
            .first()
        )
        if not car:
            raise NotFound

        assembly_photos = CarAssemblyPhoto.objects.filter(car_id=car_id).all()

        serializer_context = self.get_serializer_context()
        serializer_context['photos_serialized'] = [
            CarAssemblyPhotoSerializer(x).data for x in assembly_photos
        ]

        stages_serialized = [
            CarAssemblyStageSerializer(x, context=serializer_context).data
            for x in car.model.assembly_stages.all()
        ]

        return Response(stages_serialized)


class CarListTransportFeedView(ListModelMixin, AdminAPIView):

    serializer_class = SimpleCarSerializer
    permission_classes = []

    def get_serializer_context(self):
        context = super().get_serializer_context()

        tariff_picker = CarsharingTariffPicker()
        ride_tariff, parking_tariff = tariff_picker.pick_ride_and_parking_tariffs(
            dt=timezone.now(),
        )

        context.update(
            {
                'cost_per_minute': {
                    'carsharing_ride': ride_tariff.cost_per_minute,
                    'carsharing_parking': parking_tariff.cost_per_minute,
                },
            }
        )

        return context

    def do_get(self, request):  # pylint: disable=unused-argument
        return Response({'cars': []})

    def _check_acl(self, user):
        return True


class CarDetailsView(RetrieveModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    lookup_url_kwarg = 'car_id'
    serializer_class = CarSerializer

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        return (
            Car.objects
            .select_related('location', 'model', 'telematics_state')
        )

    def do_get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context


class CarModelListView(ListModelMixin, AdminAndCallcenterAPIView):

    serializer_class = CarModelSerializer

    def get_queryset(self):
        return CarModel.objects.all()

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class UpdateFuelCardNumberView(AdminAndCallcenterAPIView):

    lookup_url_kwarg = 'car_id'
    arguments_serializer_class = UpdateFuelCardArgumentsSerializer
    serializer_class = CarSerializer

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def get_queryset(self):
        return Car.objects

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        updater = CarUpdater(car=self.get_object())
        updater.update_fuel_card_number(request.arguments['number'])
        return Response(
            self.serializer_class(self.get_object(), context=self.get_serializer_context()).data
        )


class ActivateFuelCardView(AdminAPIView):

    lookup_url_kwarg = 'car_id'
    serializer_class = CarSerializer

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def get_queryset(self):
        return Car.objects

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        car = Car.objects.get(id=kwargs['car_id'])

        order = (
            Order.objects
            .filter(
                items__carsharing_reservation__car=car,
                completed_at__isnull=True,
            )
            .order_by('-created_at')
            .first()
        )

        processor = FuelCardProcessor(
            operator=request.user,
            source=FuelCardActivation.Source.ADMIN_PORTAL.value,
            order=order,
            car=car,
            time_to_expiry=cars.settings.EKA['time_to_block_card'],
        )
        processor.activate_fuel_card()

        return Response(
            self.serializer_class(self.get_object(), context=self.get_serializer_context()).data
        )


class BlockFuelCardView(AdminAPIView):

    lookup_url_kwarg = 'car_id'
    serializer_class = CarSerializer

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def get_queryset(self):
        return Car.objects

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        car = Car.objects.get(id=kwargs['car_id'])

        last_activation_record = (
            FuelCardActivation.objects
            .filter(
                car_id=car.id,
                blocked_at__isnull=True,
            )
            .first()
        )

        if last_activation_record:
            processor = FuelCardProcessor.from_activation_record(
                operator=request.user,
                source=FuelCardActivation.Source.ADMIN_PORTAL.value,
                activation_record=last_activation_record,
            )
        else:
            processor = FuelCardProcessor(
                operator=request.user,
                source=FuelCardActivation.Source.ADMIN_PORTAL.value,
                car=car,
                time_to_expiry=cars.settings.EKA['time_to_block_card'],
            )

        processor.block_fuel_card()

        return Response(
            self.serializer_class(self.get_object(), context=self.get_serializer_context()).data
        )


class CarCommandView(AdminAndCallcenterAPIView):

    lookup_url_kwarg = 'car_id'

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def get_queryset(self):
        return Car.objects.all()

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = self.get_object()

        status = 'ok'
        error_code = None

        try:
            response = self.do_command(car)
            response.raise_for_status()
        except TelematicsApiResponse.Error as e:
            assert e.code is not None
            status = 'error'
            error_code = 'car.{}'.format(e.code)

        return Response(
            data={
                'status': status,
                'error_code': error_code,
            }
        )

    def do_command(self, car):
        raise NotImplementedError


class OpenCarCommandView(CarCommandView):
    def do_command(self, car):
        return self.telematics_proxy.start_lease(car.imei)

class CloseCarCommandView(CarCommandView):
    def do_command(self, car):
        response = self.telematics_proxy.end_lease(car.imei)
        try:
            response.raise_for_status()
        except TelematicsApiResponse.Error:
            self.telematics_proxy.close(car.imei)
        return response

class LockHoodCommandView(CarCommandView):
    def do_command(self, car):
        return self.telematics_proxy.lock_hood(car.imei)

class UnlockHoodCommandView(CarCommandView):
    def do_command(self, car):
        return self.telematics_proxy.unlock_hood(car.imei)


class CarStatusesListView(AdminAndCallcenterAPIView):
    """
    Endpoint for SaaS team in order to detect cars in SERVICE state.
    This is needed in order to distinguish two cases:
        - an available car is moving
        - a service car is moving
    ChangeLog endpoint in ./changelog.py doesn't enable to do that, because
    it only has switches between riding statuses and AVAILABLE.
    """

    permission_classes = []

    def get_queryset(self):
        return Car.objects

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        cars_with_statuses = self.get_queryset().values('id', 'imei', 'status')
        return Response(cars_with_statuses)

    def _check_acl(self, user):
        return True


class CarStatusView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_car_available')]

    lookup_url_kwarg = 'car_id'

    allowed_source_statuses = {
        Car.Status.AVAILABLE,
        Car.Status.CLEANING,
        Car.Status.FUELING,
        Car.Status.NEW,
        Car.Status.SERVICE,
    }

    def get_queryset(self):
        return Car.objects.all()

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = self.get_object()
        car_updater = CarUpdater(car)
        try:
            self.do_update_status(car_updater)
        except car_updater.BadStatusError:
            return Response(status=HTTP_400_BAD_REQUEST)
        return Response()

    def do_update_status(self, car_updater):
        raise NotImplementedError

class MoveToAvailableCarStatusView(CarStatusView):
    def do_update_status(self, car_updater):
        car_updater.update_status(
            Car.Status.AVAILABLE,
            allowed_source_statuses=self.allowed_source_statuses,
        )

class MoveToServiceCarStatusView(CarStatusView):
    def do_update_status(self, car_updater):
        car_updater.update_status(
            Car.Status.SERVICE,
            allowed_source_statuses=self.allowed_source_statuses,
        )


class CarTagPhotosView(AdminAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    def get_queryset(self):
        return Car.objects.all()

    def do_get(self, request, *args, **kwargs):
        photos = CarsharingTagPhoto.objects.filter(tag_id=kwargs['tag_id'])

        return Response(
            CarsharingTagPhotoSerializer(photos, many=True).data,
        )

