from rest_framework.response import Response
from rest_framework.status import HTTP_204_NO_CONTENT

from cars.carsharing.core.tariff_manager import CarsharingTariffManager
from cars.carsharing.models.tariff_plan import CarsharingTariffPlan
from ..permissions import AdminPermissionCode, AdminPermissionFactory
from ..serializers.carsharing_tariffs import (
    CarsharingTariffPlanArgumentsSerializer, CarsharingTariffPlanSerializer,
)
from .base import AdminAPIView


class CarsharingTariffPlanBaseView(AdminAPIView):

    _tariff_manager = None

    @classmethod
    def _get_tariff_manager(cls):
        if cls._tariff_manager is None:
            cls._tariff_manager = CarsharingTariffManager.from_settings()
        return cls._tariff_manager


class CarsharingTariffPlanDetailsView(CarsharingTariffPlanBaseView):

    lookup_url_kwarg = 'tariff_plan_id'
    queryset = CarsharingTariffPlan.objects.all()

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_TARIFFS)]

    def get_arguments_serializer_class(self):
        if self.request.method == 'PUT':
            serializer_class = CarsharingTariffPlanArgumentsSerializer
        else:
            serializer_class = None
        return serializer_class

    def do_get(self, request, tariff_plan_id):  # pylint: disable=unused-argument
        tariff_plan = self.get_object()
        data = CarsharingTariffPlanSerializer(tariff_plan).data
        return Response(data)

    def do_put(self, request, tariff_plan_id):  # pylint: disable=unused-argument
        tariff_plan = self.get_object()
        manager = self._get_tariff_manager()

        error = None
        try:
            tariff_plan = manager.update_tariff_plan(
                tariff_plan=tariff_plan,
                name=request.arguments['name'],
                entries=request.arguments['entries'],
                car_model_code=request.arguments.get('car_model_code'),
                user_tag=request.arguments.get('user_tag'),
            )
        except manager.Error as e:
            error = str(e)

        if error is None:
            data = {
                'status': 'success',
                'tariff_plan': CarsharingTariffPlanSerializer(tariff_plan).data,
            }
        else:
            data = {
                'status': 'errors',
                'errors': [error],
            }

        return Response(data)

    def do_delete(self, request, tariff_plan_id):  # pylint: disable=unused-argument
        tariff_plan = self.get_object()
        tariff_plan.delete()
        return Response(status=HTTP_204_NO_CONTENT)


class CarsharingTariffPlanListView(CarsharingTariffPlanBaseView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_TARIFFS)]

    def get_arguments_serializer_class(self):
        if self.request.method == 'POST':
            serializer_class = CarsharingTariffPlanArgumentsSerializer
        else:
            serializer_class = None
        return serializer_class

    def do_get(self, request):  # pylint: disable=unused-argument
        results = (
            CarsharingTariffPlan.objects
            .select_related('created_by')
            .prefetch_related('entries')
        )
        data = {
            'results': [
                CarsharingTariffPlanSerializer(tariff_plan).data for tariff_plan in results
            ],
        }
        return Response(data)

    def do_post(self, request):
        manager = self._get_tariff_manager()

        error = None
        try:
            tariff_plan = manager.create_tariff_plan(
                created_by=request.user,
                name=request.arguments['name'],
                entries=request.arguments['entries'],
                car_model_code=request.arguments.get('car_model_code'),
                user_tag=request.arguments.get('user_tag'),
            )
        except manager.Error as e:
            error = str(e)

        if error is None:
            data = {
                'status': 'success',
                'tariff_plan': CarsharingTariffPlanSerializer(tariff_plan).data,
            }
        else:
            data = {
                'status': 'errors',
                'errors': [error],
            }

        return Response(data)
