from django.db.models import F, Prefetch

from rest_framework import status
from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.django.pagination import Pagination
from cars.fines.models import AutocodeFine
from cars.orders.core.order_payment_processor import OrderPaymentProcessor
from ..serializers.fine import FineListArgumentsSerializer, FineSerializer


class FineListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_fine')]

    order_payment_processor = OrderPaymentProcessor.from_settings()

    serializer_class = FineSerializer

    pagination_class = Pagination

    arguments_serializer_class = FineListArgumentsSerializer

    def get_queryset(self):
        qs = (
            AutocodeFine.objects
            .prefetch_related(
                'photos'
            )
            .filter(charged_at__isnull=False)
            .order_by(
                F('violation_time')
                .desc()
            )
        )

        id_filter = {}
        if 'user_id' in self.request.arguments:
            id_filter['user_id'] = self.request.arguments['user_id']
        if 'car_id' in self.request.arguments:
            id_filter['car_id'] = self.request.arguments['car_id']

        if id_filter:
            qs = qs.filter(**id_filter)

        return qs

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)

    def get_serializer_context(self, *args, **kwargs):
        context = super().get_serializer_context(*args, **kwargs)
        context['order_payment_processor'] = self.order_payment_processor
        return context


class FineNotifications(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_fine')]

    def do_post(self, request, fine_id):
        fine = AutocodeFine.objects.filter(id=fine_id).first()
        if not fine:
            return Response(
                data={
                    'error': 'fine_id not found',
                },
                status=status.HTTP_404_NOT_FOUND,
            )
        fine.charge_email_sent_at = None
        fine.save()
        return Response()
