import base64
import datetime
import logging
import mimetypes
import pytils

from rest_framework.response import Response
from rest_framework.status import (
    HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND, HTTP_500_INTERNAL_SERVER_ERROR
)

from cars.carsharing.models.car import Car
from cars.core.sender import EmailNotifier
from cars.users.models.user import User
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory


LOGGER = logging.getLogger(__name__)


class NotificationViewBase(DrivePermissionAPIView):
    @staticmethod
    def _convert_date(date_string):
        return pytils.dt.ru_strftime(
            "%d %B в %H:%M",
            inflected=True,
            date=datetime.datetime.strptime(date_string, '%Y-%m-%d %H:%M')
        )

    @staticmethod
    def _check_user(user):
        if user is None:
            return Response('user not found', status=HTTP_400_BAD_REQUEST)
        if user.email is None:
            return Response('user has empty email', status=HTTP_400_BAD_REQUEST)

    @staticmethod
    def _check_args_present(args, request):
        for arg in args:
            if arg not in request.data:
                return Response('need {} param'.format(arg), status=HTTP_400_BAD_REQUEST)


class SendInnerFineNotificationView(NotificationViewBase):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_fine')]

    def do_post(self, request, user_id):
        user = User.objects.filter(id=user_id).first()
        check_result = (
            self._check_user(user)
            or self._check_args_present(
                ['car_number', 'paragraph', 'sum_to_pay', 'violation_date'],
                request,
            )
        )
        if check_result:
            return check_result

        try:
            violation_date = self._convert_date(request.data['violation_date'])
        except Exception:
            return Response('bad date', status=HTTP_400_BAD_REQUEST)

        car = Car.objects.filter(number=request.data['car_number']).first()
        if car is None:
            return Response('car not found', status=HTTP_404_NOT_FOUND)
        if car.registration_id is None:
            return Response('car doesnt have STS', status=HTTP_404_NOT_FOUND)

        paragraph = request.data['paragraph']
        paragraph_link = 'https://yandex.ru/legal/drive_agreement/#index__{}'.format(
            paragraph.replace('.', '_')
        )

        sum_to_pay = request.data['sum_to_pay']

        email_notifier = EmailNotifier.from_settings()

        attachments = []
        for name, f in request.FILES.items():
            parts = name.split('.')
            if not parts:
                return Response('file needs extension', status=HTTP_400_BAD_REQUEST)
            ext = '.{}'.format(parts[-1])
            mime_type = mimetypes.types_map.get(ext)

            if not mime_type:
                return Response(
                    'unknown mime type for extension {}'.format(ext),
                    status=HTTP_400_BAD_REQUEST
                )
            attachments.append(
                {
                    "filename": name,
                    "mime_type": mime_type,
                    "data": base64.b64encode(f.read()),
                }
            )

        try:
            email_notifier.send(
                campaign='inner_fine_notification',
                to_email=user.email,
                args={
                    'sum_to_pay': sum_to_pay,
                    'paragraph': paragraph,
                    'paragraph_link': paragraph_link,
                    'car_plate': car.number,
                    'car_model': car.model.name,
                    'violation_date': violation_date,
                    'evacuation_date': 'foo',
                },
                attachments=attachments,
            )
        except Exception as e:
            LOGGER.exception('failed to send notification')

            return Response(
                'failed to send notification',
                status=HTTP_500_INTERNAL_SERVER_ERROR,
            )
        return Response()


class SendEvacuationNotificationView(NotificationViewBase):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_eva')]

    def do_post(self, request, user_id):
        user = User.objects.filter(id=user_id).first()
        check_result = (
            self._check_user(user)
            or self._check_args_present(
                ['car_number', 'receipt_number', 'sum_to_pay', 'evacuation_date'],
                request,
            )
        )
        if check_result:
            return check_result

        try:
            evacuation_date = self._convert_date(request.data['evacuation_date'])
        except Exception:
            return Response('bad evacuation date', status=HTTP_400_BAD_REQUEST)

        car = Car.objects.filter(number=request.data['car_number']).first()
        if car is None:
            return Response('car not found', status=HTTP_404_NOT_FOUND)
        if car.registration_id is None:
            return Response('car doesnt have STS', status=HTTP_404_NOT_FOUND)

        payment_link = 'https://mos.ru/shtrafy/result?sts={}'.format(car.registration_id)

        sum_to_pay = request.data['sum_to_pay']
        receipt_number = request.data['receipt_number']

        email_notifier = EmailNotifier.from_settings()
        try:
            email_notifier.send(
                campaign='evacuation_notification',
                to_email=user.email,
                args={
                    'sum_to_pay': sum_to_pay,
                    'receipt_number': receipt_number,
                    'car_plate': car.number,
                    'car_model': car.model.name,
                    'evacuation_date': evacuation_date,
                    'payment_link': payment_link,
                },
            )
        except Exception as e:
            LOGGER.exception('failed to send notification')
            return Response(
                'failed to send notification',
                status=HTTP_500_INTERNAL_SERVER_ERROR,
            )
        return Response()
