import logging
from abc import abstractmethod

from django.db import transaction
from rest_framework.mixins import ListModelMixin, RetrieveModelMixin
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND

import cars.settings
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.billing.models.card_payment import CardPayment
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.util import import_class
from cars.django.pagination import Pagination
from cars.orders.core.order_manager import OrderManager
from cars.orders.core.order_payment_processor import OrderPaymentProcessor
from cars.orders.models import Order, OrderItem, OrderItemPayment
from ..core.push_client import PUSH_CLIENT
from ..serializers.order import OrderSerializer, OrderListArgumentsSerializer


LOGGER = logging.getLogger(__name__)


class OrderFinishBaseView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    manager = OrderManager.from_settings(push_client=PUSH_CLIENT)
    order_payment_processor = OrderPaymentProcessor.from_settings()

    def do_post(self, *args, **kwargs):  # pylint: disable=unused-argument
        order_id = kwargs.get('order_id')

        order = Order.objects.filter(id=order_id).first()
        if not order:
            return Response(status=HTTP_404_NOT_FOUND)

        if order.completed_at is not None:
            return Response(
                {
                    'reason': 'order already completed',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        finished_order = OrderSerializer(
            self._do_finish_order_action(order),
            context=self.get_serializer_context(),
        ).data

        return Response(finished_order)

    def get_serializer_context(self, *args, **kwargs):
        context = super().get_serializer_context(*args, **kwargs)
        context['order_payment_processor'] = self.order_payment_processor
        return context

    @abstractmethod
    def _do_finish_order_action(self, order):
        raise NotImplementedError


class OrderFinishView(OrderFinishBaseView):

    def _do_finish_order_action(self, order):
        return self.manager.force_complete_order(order)


class OrderFinishNoTelematicsView(OrderFinishBaseView):

    def _do_finish_order_action(self, order):
        return self.manager.force_complete_order_ignore_telematics(order)


class OrderListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    pagination_class = Pagination
    serializer_class = OrderSerializer

    arguments_serializer_class = OrderListArgumentsSerializer

    saas_client = SaasDriveAdminClient.from_settings()
    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        if not self.saas_client.check_access_to_deleting_user(
                self.request.arguments.get('user_id', None), self.request
        ):
            return Order.objects.none()

        qs = (
            Order.objects
            .with_related()
            .using(cars.settings.DB_RO_ID)
            .select_related('user__registration_state')
            .order_by('-created_at')
        )

        filter_dict = {}
        if 'user_id' in self.request.arguments:
            filter_dict['user_id'] = self.request.arguments['user_id']
        if 'car_id' in self.request.arguments:
            filter_dict['items__carsharing_reservation__car_id'] = self.request.arguments['car_id']
        if 'created_at_gte' in self.request.arguments:
            filter_dict['created_at__gte'] = self.request.arguments['created_at_gte']
        if 'created_at_lte' in self.request.arguments:
            filter_dict['created_at__lte'] = self.request.arguments['created_at_lte']
        if 'completed_at_gte' in self.request.arguments:
            filter_dict['completed_at__gte'] = self.request.arguments['completed_at_gte']
        if 'completed_at_lte' in self.request.arguments:
            filter_dict['completed_at__lte'] = self.request.arguments['completed_at_lte']
        if 'is_completed' in self.request.arguments:
            filter_dict['completed_at__isnull'] = not self.request.arguments['is_completed']

        if filter_dict:
            qs = qs.filter(**filter_dict)

        return qs

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class OrderDetailsView(RetrieveModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    lookup_url_kwarg = 'order_id'
    serializer_class = OrderSerializer

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        return Order.objects.with_related()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def do_get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)


class RefundOrderView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    lookup_url_kwarg = 'order_id'

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        return Order.objects.with_related()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def do_post(self, request, order_id):  # pylint: disable=unused-argument
        order = self.get_object()
        self.order_payment_processor.refund_order(order)

        order = self.get_object()  # Sync with DB.
        data = OrderSerializer(order, context=self.get_serializer_context()).data

        return Response(data)


class RefundOrderItemView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    lookup_url_kwarg = 'order_item_id'

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        return OrderItem.objects.with_related()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['order_payment_processor'] = self.order_payment_processor
        return context

    def do_post(self, request, order_id, order_item_id):  # pylint:disable=unused-argument
        order_item = self.get_object()
        self.order_payment_processor.refund_order_item(order_item)

        order = Order.objects.get(id=order_item.order.id)
        data = OrderSerializer(order, context=self.get_serializer_context()).data

        return Response(data)


class TrustProxyPaymentView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_payment')]

    trust_client = (
        import_class(cars.settings.TRUST['client_class']).from_settings(push_client=PUSH_CLIENT)
    )

    def do_get(self, request, purchase_token):  # pylint: disable=unused-argument
        payment = (
            CardPayment.objects
            .select_related('user')
            .filter(purchase_token=purchase_token)
            .first()
        )

        if payment is not None:
            uid = payment.user.uid
        else:
            uid = request.query_params.get('uid')

        if uid is None:
            return Response(status=HTTP_404_NOT_FOUND)

        data = self.trust_client.get_payment(
            uid=uid,
            purchase_token=purchase_token,
        )

        return Response(data)


class OrderRestartPaymentView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    lookup_url_kwarg = 'order_id'

    order_payment_processor = OrderPaymentProcessor.from_settings()

    def get_queryset(self):
        return OrderItemPayment.objects

    def do_post(self, request, order_id):  # pylint:disable=unused-argument
        payments = (
            self.get_queryset()
            .filter(
                order_item__order_id=order_id,
                card_payment__status=CardPayment.Status.DRAFT.value
            )
        )

        with transaction.atomic():
            for payment in payments:
                payment.delete()

        return Response()
