import logging

from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response
from rest_framework.status import (
    HTTP_400_BAD_REQUEST,
    HTTP_404_NOT_FOUND,
)

from cars.admin.models.push_preset import PushPreset
from cars.admin.permissions import AdminPermissionFactory, AdminPermissionCode
from cars.admin.serializers.push_preset import PushPresetSerializer
from .base import AdminAPIView


LOGGER = logging.getLogger(__name__)


class PushPresetListView(ListModelMixin, AdminAPIView):
    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.ACCESS),
    ]
    serializer_class = PushPresetSerializer

    def get_queryset(self):
        return PushPreset.objects

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class PushPresetModifyView(AdminAPIView):

    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.ACCESS),
        AdminPermissionFactory.build(AdminPermissionCode.OPERATE_PUSH_PRESETS)
    ]

    def do_post(self, request):
        try:
            message = request.data['message']
        except KeyError:
            return Response(status=HTTP_400_BAD_REQUEST)
        p = PushPreset.objects.create(message=message)
        return Response({
            'id': str(p.id)
        })

    def do_delete(self, request, preset_id):
        PushPreset.objects.filter(id=preset_id).delete()
        return Response()

    def do_put(self, request, preset_id):
        p = PushPreset.objects.filter(id=preset_id).first()
        if p is None:
            return Response(status=HTTP_404_NOT_FOUND)
        try:
            p.message = request.data['message']
            p.save()
        except KeyError:
            return Response(status=HTTP_400_BAD_REQUEST)
        return Response()
