import time

from django.db import transaction
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.registration.core.registration_manager import RegistrationManager
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.core.user_profile_updater import UserHistoryManager
from cars.users.models import User, UserDocumentPhoto
from ..serializers.registration import RegistrationResubmitPhotosArgumentsSerializer
from ..serializers.user import UserSerializer
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory


class UserRegistrationView(DrivePermissionAPIView):

    lookup_url_kwarg = 'user_id'

    datasync = DataSyncDocumentsClient.from_settings()
    registration_manager = RegistrationManager.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['datasync_client'] = self.datasync
        return context

    def get_queryset(self):
        return (
            User.objects
            .select_related('registration_state')
            .prefetch_related(
                'documents__photos__background_video',
            )
        )

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user = self.get_object()

        try:
            user = self.register(user)
        except self.get_bad_request_errors() as e:
            return Response(
                status=HTTP_400_BAD_REQUEST,
                data={
                    'errors': [str(e)],
                },
            )

        data = UserSerializer(user, context=self.get_serializer_context()).data

        return Response(data)

    def get_bad_request_errors(self):
        return (
            self.registration_manager.AlreadyRegisteredError,
            self.registration_manager.ChatNotCompletedError,
        )

    def register(self, user):
        raise NotImplementedError


class RegistrationApproveView(UserRegistrationView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def register(self, user):
        return self.registration_manager.approve(user, force=True)


class RegistrationApproveRejectedView(UserRegistrationView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def get_bad_request_errors(self):
        return super().get_bad_request_errors() + (
            self.registration_manager.NotRejectedError,
        )

    def register(self, user):
        return self.registration_manager.approve_rejected(user)


class RegistrationRejectView(UserRegistrationView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def register(self, user):
        return self.registration_manager.reject(user)


class RegistrationResubmitPhotosView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    lookup_url_kwarg = 'user_id'

    arguments_serializer_class = RegistrationResubmitPhotosArgumentsSerializer
    registration_manager = RegistrationManager.from_settings()

    def get_queryset(self):
        return User.objects.select_related('registration_state')

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user = self.get_object()
        photo_types = [UserDocumentPhoto.Type(x) for x in request.arguments['photo_types']]

        error = None
        try:
            self.registration_manager.resubmit_photos(user=user, photo_types=photo_types)
        except self.registration_manager.ChatNotCompletedError:
            error = 'chat.incomplete'
        except self.registration_manager.PhotoResubmitInProgressError:
            error = 'resubmit.inprogress'

        if error:
            return Response(
                status=HTTP_400_BAD_REQUEST,
                data={
                    'errors': [error],
                },
            )

        return Response()


class RegistrationResetDebugView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    lookup_url_kwarg = 'user_id'

    history_manager = UserHistoryManager()

    def get_queryset(self):
        return User.objects.select_related('registration_state')

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user = self.get_object()
        registration_state = user.get_registration_state()

        if registration_state is not None:
            registration_state.outcome_notification_sent_at = None

        user.status = User.Status.ONBOARDING.value

        with transaction.atomic(savepoint=False):
            registration_state.save()

            user.updated_at = time.time()
            user.save()

            self.history_manager.update_entry(user, str(request.user.id))

        return Response()
