import logging
from uuid import UUID

from rest_framework.response import Response
from rest_framework.status import (
    HTTP_400_BAD_REQUEST,
    HTTP_404_NOT_FOUND,
    HTTP_500_INTERNAL_SERVER_ERROR
)

from cars.admin.core.send_push_manager import SendPushManager
from cars.admin.models.push_preset import PushPreset
from cars.admin.permissions import AdminPermissionFactory, AdminPermissionCode
from .base import AdminAPIView
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory


LOGGER = logging.getLogger(__name__)


class UserPushView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_push')]

    def do_post(self, request, user_id):
        try:
            message = request.data['message']
        except Exception:
            LOGGER.exception('Bad push request')
            return Response(status=HTTP_400_BAD_REQUEST)

        sender = request.user

        push_manager = SendPushManager.from_settings()

        try:
            push_manager.send_push(user_id=user_id, message=message, sender=sender)
        except AssertionError:
            return Response(status=HTTP_404_NOT_FOUND)
        except Exception:
            LOGGER.exception('Failed to send push notification')
            return Response(status=HTTP_500_INTERNAL_SERVER_ERROR)

        return Response()


class UserPushPresetView(AdminAPIView):

    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.ACCESS),
        AdminPermissionFactory.build(AdminPermissionCode.SEND_PRESET_PUSH)
    ]

    def do_post(self, request, user_id):
        try:
            preset_id = request.data['id']
        except KeyError:
            return Response(status=HTTP_400_BAD_REQUEST)

        preset = PushPreset.objects.filter(id=preset_id).first()
        if preset is None:
            return Response(status=HTTP_404_NOT_FOUND)
        message = preset.message

        sender = request.user

        push_manager = SendPushManager.from_settings()

        try:
            push_manager.send_push(user_id=user_id, message=message, sender=sender)
        except AssertionError:
            return Response(status=HTTP_404_NOT_FOUND)
        except Exception:
            LOGGER.exception('Failed to send push notification')
            return Response(status=HTTP_500_INTERNAL_SERVER_ERROR)

        return Response()


class UserListPushView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_push')]

    def do_post(self, request):
        try:
            message = request.data['message']
            ttl = request.data.get('ttl', None)
            lines = [x.strip() for x in request.data['user_ids_file']]
            user_ids = [x if isinstance(x, str) else x.decode('ascii')
                        for x in lines if x]
            logging.info('push batch handler got user ids: {}'.format(user_ids))
            for user_id in user_ids:
                try:
                    UUID(user_id)
                except ValueError:
                    LOGGER.exception('push request got bad uuid value {}'.format(user_id))
                    raise
        except Exception:
            LOGGER.exception('Bad push request')
            return Response(status=HTTP_400_BAD_REQUEST)

        sender = request.user

        push_manager = SendPushManager.from_settings()

        try:
            return Response(
                push_manager.send_push_to_batch(user_ids, message, ttl, sender)
            )

        except Exception:
            LOGGER.exception('Failed to send push to batch')
            return Response(status=HTTP_500_INTERNAL_SERVER_ERROR)


class UserListPushPresetView(AdminAPIView):

    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.ACCESS),
        AdminPermissionFactory.build(AdminPermissionCode.OPERATE_PUSH_PRESETS)
    ]

    def do_post(self, request):
        try:
            preset_id = request.data['id']
        except KeyError:
            return Response(status=HTTP_400_BAD_REQUEST)

        preset = PushPreset.objects.filter(id=preset_id).first()
        if preset is None:
            return Response(status=HTTP_404_NOT_FOUND)
        message = preset.message

        try:
            ttl = request.data.get('ttl', None)
            lines = [x.strip() for x in request.data['user_ids_file']]
            user_ids = [x if isinstance(x, str) else x.decode('ascii')
                        for x in lines if x]
            logging.info('push batch handler got user ids: {}'.format(user_ids))
            for user_id in user_ids:
                try:
                    UUID(user_id)
                except ValueError:
                    LOGGER.exception('push request got bad uuid value {}'.format(user_id))
                    raise
        except Exception:
            LOGGER.exception('Bad push request')
            return Response(status=HTTP_400_BAD_REQUEST)

        sender = request.user

        push_manager = SendPushManager.from_settings()

        try:
            return Response(
                push_manager.send_push_to_batch(user_ids, message, ttl, sender)
            )
        except Exception:
            LOGGER.exception('Failed to send push to batch')
            return Response(status=HTTP_500_INTERNAL_SERVER_ERROR)
