import logging

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_422_UNPROCESSABLE_ENTITY, HTTP_500_INTERNAL_SERVER_ERROR

import cars.settings

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.core.telephony import BadTelephonyApiResponseError
from cars.callcenter.core import (
    CallPriorityManager,
    CallRegistrar,
    PhoneBlacklistManager,
    StaffInfoHelper,
    StaffStatAccessHelper,
    LoadBalanceHelper,
)
from cars.callcenter.serializers.user import StaffEntrySerializer

from cars.request_aggregator.core import CallRegisteringWebHookHelper, SolomonHelper

from ..serializers.telephony import (
    TelephonyAssignedCallArgumentsSerializer,
    TelephonyRegisterCallArgumentsSerializer,
    TelephonyOperatorsArgumentsSerializer,
    BlacklistAddPhoneArgumentsSerializer,
    BlacklistModificationArgumentsSerializer,
    BlacklistHistoryArgumentsSerializer,
    BlacklistHistorySerializer,
    PriorityUserArgumentsSerializer,
    LoadBalanceArgumentsSerializer,
)

from .base import AdminAPIView


LOGGER = logging.getLogger(__name__)


class SimplifiedRegisterCallView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_data_routing')]

    arguments_serializer_class = TelephonyRegisterCallArgumentsSerializer

    call_registrar = CallRegistrar.from_settings()

    def do_post(self, request, action):
        from_number = request.arguments['from_number']
        to_number = request.arguments['to_number']

        response_message = response_status = None

        try:
            self.call_registrar.register_call(
                action=CallRegistrar.Action(action),
                from_number=from_number,
                to_number=to_number,
            )
        except RuntimeError as e:
            LOGGER.exception('call assignment runtime error')
            response_message = str(e)
            response_status = HTTP_500_INTERNAL_SERVER_ERROR
        except Exception:
            LOGGER.exception('call assignment internal error')
            response_message = 'internal error'
            response_status = HTTP_500_INTERNAL_SERVER_ERROR

        response = Response(response_message, status=response_status)
        return response


class GenericRegisterCallView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_data_routing')]

    call_registrar = CallRegisteringWebHookHelper.from_settings()
    solomon_helper = SolomonHelper('call_registration_webhook')

    def do_post(self, request):
        try:
            processed_entries = self.call_registrar.process_data(request.data)
        except Exception as exc:
            return Response(data=str(exc), status=HTTP_400_BAD_REQUEST)

        response_data = None
        response_status = None

        try:
            entries_stat = self.call_registrar.register_entries(processed_entries)
        except Exception:
            total_processed_entries_count = sum(len(x) for x in processed_entries)
            entries_stat = {'error': total_processed_entries_count}

            response_status = HTTP_500_INTERNAL_SERVER_ERROR
            response_data = 'call registering internal error'
            LOGGER.exception(response_data)
        else:
            total_processed_entries_count = sum(entries_stat.values())
            response_data = 'registered {} entries'.format(total_processed_entries_count)

        self.solomon_helper.increment_counters(**entries_stat)

        response = Response(response_data, status=response_status)
        return response


class AssignedCallView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    arguments_serializer_class = TelephonyAssignedCallArgumentsSerializer

    call_registrar = CallRegistrar.from_settings()

    def do_get(self, request):
        # poll_call_assignment can be used
        assignment_data = self.call_registrar.get_call_assignment(request.user, actual_data=request.arguments)
        return Response(assignment_data)


class AllAssignedCallsView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    call_registrar = CallRegistrar.from_settings()

    def do_get(self, request):
        assignment_data = self.call_registrar.get_all_call_assignments(request)
        return Response(assignment_data)


class AllOperatorsView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    arguments_serializer_class = TelephonyOperatorsArgumentsSerializer

    staff_info_helper = StaffInfoHelper.make_default()

    def do_get(self, request):
        department_url_prefix = request.arguments['department_url_prefix'] or None
        entries = self.staff_info_helper.get_all_staff_entries(department_url_prefix=department_url_prefix)
        formatted_entries = [StaffEntrySerializer(e).data for e in entries]
        response_data = {'data': formatted_entries, 'count': len(entries)}
        return Response(response_data)


class OperatorsWithoutStatAccessView(AdminAPIView):

    arguments_serializer_class = TelephonyOperatorsArgumentsSerializer

    staff_stat_access_helper = StaffStatAccessHelper()

    def do_get(self, request):
        department_url_prefix = request.arguments['department_url_prefix'] or None
        entries = self.staff_stat_access_helper.get_entries_without_access(department_url_prefix=department_url_prefix)
        return Response(entries)


class AllStatAccessOperatorsView(AdminAPIView):

    arguments_serializer_class = TelephonyOperatorsArgumentsSerializer

    staff_stat_access_helper = StaffStatAccessHelper()

    def do_get(self, request):
        department_url_prefix = request.arguments['department_url_prefix'] or None
        entries = self.staff_stat_access_helper.get_all_entries(department_url_prefix=department_url_prefix)
        return Response(entries)


class PriorityUserView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    arguments_serializer_class = PriorityUserArgumentsSerializer

    helper = CallPriorityManager.from_settings()

    def do_get(self, request):
        phone = str(request.arguments['phone'])
        is_prioritized = self.helper.is_user_prioritized(phone=phone)
        response_data = {'phone': phone, 'is_prioritized': is_prioritized}
        return Response(response_data)


class AllPriorityUsersView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    helper = CallPriorityManager.from_settings()

    def do_get(self, request):
        entries = self.helper.get_prioritized_formatted_user_entries()
        response_data = {'data': entries, 'count': len(entries)}
        return Response(response_data)


class CallBlacklistAddPhoneView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    helper = PhoneBlacklistManager.from_settings()

    def do_post(self, request):
        serializer = BlacklistAddPhoneArgumentsSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        try:
            self.helper.add_phone_to_blacklist(request.user, **serializer.validated_data)
        except Exception as exc:
            LOGGER.exception('error adding phone to blacklist')
            return Response(str(exc), status=HTTP_400_BAD_REQUEST)

        return Response()


class CallBlacklistRemovePhoneView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    helper = PhoneBlacklistManager.from_settings()

    def do_post(self, request):
        serializer = BlacklistModificationArgumentsSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        try:
            self.helper.remove_phone_from_blacklist(request.user, **serializer.validated_data)
        except Exception as exc:
            LOGGER.exception('error removing phone from blacklist')
            return Response(str(exc), status=HTTP_400_BAD_REQUEST)

        return Response()


class CallBlacklistCheckPhoneView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    arguments_serializer_class = BlacklistModificationArgumentsSerializer

    helper = PhoneBlacklistManager.from_settings()

    def do_get(self, request):
        phone = str(request.arguments['phone'])
        is_blacklisted, blocked_until = self.helper.is_phone_blacklisted(phone=phone)
        blocked_until = blocked_until.timestamp() if blocked_until is not None else None
        response_data = {'phone': phone, 'is_blacklisted': is_blacklisted, 'blocked_until': blocked_until}
        return Response(response_data)


class CallBlacklistHistoryView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    arguments_serializer_class = BlacklistHistoryArgumentsSerializer

    helper = PhoneBlacklistManager.from_settings()

    def do_get(self, request):
        entries = self.helper.get_phone_blacklist_history(**request.arguments)
        formatted_entries = [BlacklistHistorySerializer(e).data for e in entries]
        response_data = {'data': formatted_entries, 'count': len(entries)}
        return Response(response_data)


class CallCenterLoadBalanceCalendarView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_manager')]

    load_balance_settings_helper = LoadBalanceHelper.from_settings()

    arguments_serializer_class = LoadBalanceArgumentsSerializer

    def do_get(self, request):
        application_city = request.arguments['city']
        application_source = cars.settings.CALLCENTER['distribute_apps'][application_city]
        settings = self.load_balance_settings_helper.get_calendar_values(application_source)
        return Response(
            {'city': application_city, 'data': settings}
        )

    def do_post(self, request):
        performer = request.user

        application_city = request.arguments['city']
        application_source = cars.settings.CALLCENTER['distribute_apps'][application_city]

        settings = request.data.get('data', None)

        if settings is None:
            return Response(
                'call center load balance settings do not exist or have incorrect format',
                status=HTTP_400_BAD_REQUEST
            )

        try:
            self.load_balance_settings_helper.check_calendar_values(settings)
        except Exception as exc:
            return Response(str(exc), status=HTTP_400_BAD_REQUEST)

        try:
            updated_settings = self.load_balance_settings_helper.set_calendar_values(
                application_source, settings, performer
            )
        except BadTelephonyApiResponseError as exc:
            return Response(status=HTTP_422_UNPROCESSABLE_ENTITY, data={'error': str(exc)})

        return Response(
            {'city': application_city, 'data': updated_settings}
        )
