import logging
import time

from django.db import transaction
from django.db.models import F, Q
from rest_framework.mixins import ListModelMixin, RetrieveModelMixin
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND
from rest_framework.response import Response

import cars.settings
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.django.pagination import Pagination
from cars.registration.core import ChatManager
from cars.registration.core.registration_manager import RegistrationManager
from cars.users.core.user_profile_updater import UserProfileUpdater, UserHistoryManager
from cars.users.models import User, UserPhoneBinding
from cars.drive.core.plus_status_updater import IntroScreensUpdater
from ..permissions import AdminPermissionCode, AdminPermissionFactory
from ..serializers.user import (
    UserPhoneHistoryViewArgumentsSerializer, UserPhoneHistoryViewSerializer,
    UserListArgumentsSerializer, UserSerializer,
    UserCreateViewArgumentsSerializer, UserChatProxyViewArgumentsSerializer)
from ..serializers.user_simple import SimpleUserSerializer
from .base import AdminAndCallcenterAPIView, AdminAPIView

LOGGER = logging.getLogger(__name__)


class UserPhoneSetUnverifiedView(RetrieveModelMixin, AdminAndCallcenterAPIView):  # pylint: disable=line-too-long

    serializer_class = UserSerializer
    lookup_url_kwarg = 'user_id'

    def get_queryset(self):
        return User.objects

    def do_post(self, request, *args, **kwargs):  # pylint:disable=unused-argument
        instance = self.get_object()
        updater = UserProfileUpdater(instance)
        updater.request_phone_verification(request.user.id)
        return Response()


# temporary phone change history view
class UserPhoneHistoryView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    MAX_COUNT = 100

    arguments_serializer_class = UserPhoneHistoryViewArgumentsSerializer

    def do_get(self, request):
        user_id = request.arguments['user_id']

        user_status_qs = User.objects.filter(id=user_id).values_list('status', flat=True)
        user_status = next(iter(user_status_qs), None)

        if user_status is None or user_status == 'deleted':
            return Response(status=HTTP_404_NOT_FOUND)

        entries = UserPhoneBinding.objects.filter(user_id=user_id).order_by('-submit_date')[:self.MAX_COUNT]
        formatted_entries = [UserPhoneHistoryViewSerializer(e).data for e in entries]
        return Response(
            data={'data': formatted_entries, 'count': len(formatted_entries)}
        )


from cars.django.views import CarsharingAPIView
class UserDetailsView(RetrieveModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    lookup_url_kwarg = 'user_id'
    serializer_class = UserSerializer

    saas_client = SaasDriveAdminClient.from_settings()
    history_manager = UserHistoryManager()

    def get_queryset(self):
        return User.objects.with_related()

    def remove_personal_data(self, data):
        data['driver_license']['text_data'] = None
        if data['passport']['text_data']:
            passport_text_data = data['passport']['text_data']
            data_json_truncated_passport = {
                'first_name': passport_text_data.get('first_name', ''),
                'last_name': passport_text_data.get('last_name', ''),
                'patronymic_name': passport_text_data.get('patronymic_name', ''),
                'birth_date': passport_text_data.get('birth_date', ''),
                'birth_place': passport_text_data.get('birth_place', ''),
                'gender': passport_text_data.get('gender', ''),
            }
            data['passport']['text_data'] = data_json_truncated_passport

        data['assignments'] = None

        return data

    def do_delete(self, request, *args, **kwargs):  # pylint: disable=unused-arument
        instance = self.get_object()

        instance.status = 'deleted'
        instance.updated_at = time.time()
        instance.uid *= -1
        instance.email = '!' + str(instance.email)

        with transaction.atomic(savepoint=False):
            instance.save()
            self.history_manager.update_entry(instance, str(request.user.id))

        instance.refresh_from_db()

        serializer = self.get_serializer(instance)
        data = serializer.data

        return Response(data)

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        instance = self.get_object()
        account_info_updater = IntroScreensUpdater.from_settings_admin_api(instance)

        blackbox_info = None
        if instance.status != 'deleted':
            bb_info_denied_roles = cars.settings.USERS['view_restrictions']['blackbox_info']['denied_roles']
            if not self.saas_client.check_user_role(*bb_info_denied_roles, request=request, require_all=False):
                blackbox_info = account_info_updater.get_blackbox_info()
        else:
            instance = User()

        serializer = self.get_serializer(instance)
        serializer.context['blackbox_info'] = blackbox_info

        data = serializer.data

        return Response(data)


class UserListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    pagination_class = Pagination
    serializer_class = SimpleUserSerializer

    arguments_serializer_class = UserListArgumentsSerializer

    def get_queryset(self):
        qs = (
            User.objects
            .using(cars.settings.DB_RO_ID)
            .select_related('bonus_account', 'registration_state')
            .order_by(F('registered_at').desc(nulls_last=True))
        )

        query = self.request.arguments.get('q')
        if query:
            query_tokens = query.split()
            for query_token in query_tokens:
                lookup_fields = [
                    'first_name',
                    'last_name',
                    'patronymic_name',
                    'email',
                    'phone',
                    'username',
                ]

                dbquery = Q()
                for lookup_field in lookup_fields:
                    dbquery |= Q(**{
                        '{}__icontains'.format(lookup_field): query_token,
                    })

                qs = qs.filter(dbquery)

        if self.request.arguments.get('status'):
            qs = qs.filter(
                status=self.request.arguments['status'],
            )

        if self.request.arguments.get('tag'):
            qs = qs.filter(
                tags__contains=[self.request.arguments['tag']],
            )

        if self.request.arguments.get('chat_action'):
            qs = qs.filter(
                registration_state__chat_action_id=self.request.arguments['chat_action'],
            )

        if self.request.arguments.get('chat_completed'):
            qs = qs.filter(
                registration_state__chat_completed_at__isnull=False,
                registration_state__chat_action_id__isnull=True,
            )

        return qs

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class UserStatusListView(AdminAndCallcenterAPIView):

    def do_get(self, _):
        data = [
            {
                'id': s.value,
                'name': s.name.capitalize(),
            }
            for s in User.Status
        ]
        return Response(data)


class UserCreateView(AdminAPIView):

    arguments_serializer_class = UserCreateViewArgumentsSerializer

    history_manager = UserHistoryManager()

    def do_post(self, request, *args, **kwargs):
        uid = request.arguments['uid']
        if User.objects.filter(uid=uid).count() > 0:
            return Response(
                {
                    'error': 'user.exists',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        u = User(uid=request.arguments['uid'])
        u.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            u.save()
            self.history_manager.add_entry(u, str(request.user.id))

        return Response()


class UserChatProxyView(DrivePermissionAPIView):
    """
    PythonAPI chat client for new backend.
    """
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    arguments_serializer_class = UserChatProxyViewArgumentsSerializer

    history_manager = UserHistoryManager()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._chat_manager = ChatManager.from_settings()

    def do_post(self, request, *args, **kwargs):
        id_ = request.arguments['id']
        action = request.arguments['action']
        param = request.arguments['param']

        user = User.objects.filter(id=id_).first()
        if not user:
            return Response(
                {
                    'error': 'user.absent',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        for attempt in range(3):
            try:
                chat_session = self.get_chat_session(user=user)
                if action == 'clear_chat':
                    chat_session.clear_chat(force=True)
                elif action == 'send_message_group':
                    chat_session.send_message_group(id_=param)
                elif action == 'enqueue_chat_action':
                    chat_session.enqueue_chat_action(chat_action_id=param)
                elif action == 'block_old_license':
                    with transaction.atomic():
                        chat_session.clear_chat(force=True)
                        chat_session.send_message_group(id_='block_old_license_intro')
                        chat_session.enqueue_chat_action(chat_action_id='block_old_license_1')
                        chat_session.enqueue_chat_action(chat_action_id='block_old_license_2')
                elif action == 'block_generic':
                    with transaction.atomic():
                        chat_session.clear_chat(force=True)
                        chat_session.send_message_group(id_=param)
                else:
                    return Response(
                        {
                            'error': 'action.unknown',
                        },
                        status=HTTP_400_BAD_REQUEST,
                    )
            except Exception:
                LOGGER.exception('unable to do a db write')
                if attempt == 2:
                    raise
            else:
                break

        return Response()

    def get_chat_session(self, user):
        chat_context = {
            'user': user,
        }
        chat_session = self._chat_manager.make_session(user=user, context=chat_context)
        return chat_session
