from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response

from cars.billing.core.bonus_account_manager import BonusAccountManager
from cars.billing.models.bonus_account import BonusAccountOperation
from cars.django.pagination import Pagination
from cars.users.models.user import User
from ..permissions import AdminPermissionCode, AdminPermissionFactory
from ..serializers.user import UserSerializer
from ..serializers.user_bonus_account import (
    UserBonusAccountArgumentsSerializer, UserBonusAccountOperationSerializer,
)
from .base import AdminAPIView
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory


class UserBonusAccountView(AdminAPIView):

    lookup_url_kwarg = 'user_id'
    queryset = User.objects.with_related()

    arguments_serializer_class = UserBonusAccountArgumentsSerializer
    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_BONUS_ACCOUNTS)]

    def update_bonus_account(self, manager, operator, amount, comment, nonce):  # pylint: disable=unused-argument
        raise RuntimeError

    def do_post(self, request, user_id):  # pylint: disable=unused-argument
        user = self.get_object()

        manager = BonusAccountManager.from_user(user)

        error = None
        try:
            self.update_bonus_account(
                manager=manager,
                operator=request.user,
                amount=request.arguments['amount'],
                comment=request.arguments['comment'],
                nonce=request.arguments['nonce'],
            )
        except manager.InsufficientFundsError:
            error = 'user.bonus_account.insufficient_funds'

        if error is None:
            user.bonus_account = manager.account
            user_data = UserSerializer(user, context=self.get_serializer_context()).data
            data = {
                'status': 'success',
                'user': user_data,
            }
        else:
            data = {
                'status': 'errors',
                'errors': [error],
            }

        return Response(data)


class CreditUserBonusAccountView(UserBonusAccountView):

    def update_bonus_account(self, manager, operator, amount, comment, nonce):
        manager.credit_generic(
            operator=operator,
            amount=amount,
            comment=comment,
            nonce=nonce,
        )


class DebitUserBonusAccountView(UserBonusAccountView):

    def update_bonus_account(self, manager, operator, amount, comment, nonce):
        manager.debit_generic(
            operator=operator,
            amount=amount,
            comment=comment,
            nonce=nonce,
        )


class UserBonusAccountOperationListView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    pagination_class = Pagination
    serializer_class = UserBonusAccountOperationSerializer

    def get_queryset(self):
        return []

    def do_get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)
