from django.http import Http404
from django.shortcuts import get_object_or_404
from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.core.images.video import VideoWatermarker
from cars.core.mds.wrapper import MDSDocumentsWrapper
from cars.core.images import ImageProcessor, Resizer, Watermarker
from cars.django.parsers import ImageBinaryStreamParser, VideoBinaryStreamParser
from cars.django.renderers import JPEGRenderer, MP4Renderer
from cars.django.util import parse_uuid_or_404
from cars.django.views import StreamingResponseViewMixin
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.models import UserDocumentBackgroundVideo, UserDocumentPhoto, User
from cars.users.core.user_profile_updater import UserProfileUpdater
from ..permissions import AdminPermissionCode, AdminPermissionFactory
from ..serializers.user_documents_textdata import UserDocumentsTextDataUpdateSerializer
from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory

from .base import AdminAPIView


class BaseUserDocumentPhotoView(DrivePermissionAPIView):

    lookup_url_kwarg = 'photo_id'
    renderer_classes = [JPEGRenderer]

    mds_client = MDSDocumentsWrapper.from_settings()

    def get_queryset(self):
        if 'document_id' in self.kwargs and self.kwargs['document_id'] != 'null':
            return (
                UserDocumentPhoto.objects
                .filter(
                    user_id=self.kwargs['user_id'],
                    id=self.kwargs['photo_id'],
                )
            )
        else:
            fake_user_document_photo = UserDocumentPhoto(
                user_id=self.kwargs['user_id'],
                id=self.kwargs['photo_id']
            )
            return [fake_user_document_photo]
    
    def get_object(self):
        qs = self.get_queryset()
        if isinstance(qs, list):
            return qs[0]
        return super().get_object()


class UserDocumentPhotoContentView(BaseUserDocumentPhotoView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._watermarker_props = cars.settings.ADMIN['documents']['images']['watermarker']

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user_document_photo = self.get_object()

        watermarker = Watermarker(
            base_watermark_size=self._watermarker_props['base_watermark_size'],
            watermark_visibility=self._watermarker_props['watermark_visibility'],
            font_size=self._watermarker_props['font_size'],
            watermark_first_line=str(request.user.uid),
        )

        image_processor = ImageProcessor(
            watermarker=watermarker,
            resizer=None,
        )

        try:
            data = self.mds_client.get_user_document_photo(
                user_document_photo=user_document_photo,
                image_processor=image_processor,
            )
        except self.mds_client.ObjectDoesNotExist:
            raise NotFound
        return Response(data)


class UserDocumentPhotoThumbnailView(BaseUserDocumentPhotoView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self._watermarker_props = cars.settings.ADMIN['documents']['thumbnails']['watermarker']

        self._resizer = Resizer(
            cars.settings.ADMIN['documents']['thumbnails']['greater_side_to_fit']
        )

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user_document_photo = self.get_object()

        watermarker = Watermarker(
            base_watermark_size=self._watermarker_props['base_watermark_size'],
            watermark_visibility=self._watermarker_props['watermark_visibility'],
            font_size=self._watermarker_props['font_size'],
            watermark_first_line=str(request.user.uid),
        )

        image_processor = ImageProcessor(
            watermarker=watermarker,
            resizer=self._resizer,
        )

        try:
            data = self.mds_client.get_user_document_photo(
                user_document_photo=user_document_photo,
                image_processor=image_processor,
            )
        except self.mds_client.ObjectDoesNotExist:
            raise NotFound
        return Response(data)


class UserDocumentPhotoBackgroundVideoView(StreamingResponseViewMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    lookup_field = 'photo_id'

    mds_client = MDSDocumentsWrapper.from_settings()

    def get_queryset(self):
        if 'document_id' in self.kwargs and self.kwargs['document_id'] != 'null':
            return (
                UserDocumentBackgroundVideo.objects
                .filter(
                    photo_id=self.kwargs['photo_id'],
                )
            )
        else:
            # actually video id can differ from photo id but they are equal mostly
            bv_ids = list(
                UserDocumentBackgroundVideo.objects
                .filter(photo_id=self.kwargs['photo_id'])
                .values_list('id', flat=True)
            )
            if not bv_ids:
                raise Http404

            (bv_id, ) = bv_ids  # unique

            fake_user_document_photo = UserDocumentPhoto(
                user_id=self.kwargs['user_id'],
                id=self.kwargs['photo_id']
            )
            fake_user_document_bv = UserDocumentBackgroundVideo(
                photo=fake_user_document_photo,
                id=bv_id
                # mime-type is not specified
            )
            return [fake_user_document_bv]

    def get_object(self):
        qs = self.get_queryset()
        if isinstance(qs, list):
            return qs[0]
        return super().get_object()

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user_document_background_video = self.get_object()
        try:
            mds_response = self.mds_client.get_user_document_background_video(
                user_document_background_video,
            )
        except self.mds_client.ObjectDoesNotExist:
            raise NotFound

        stream = mds_response['Body']
        content_type = user_document_background_video.mime_type
        content_length = mds_response['ContentLength']

        return self.get_response(
            stream=stream,
            content_type=content_type,
            content_length=content_length,
        )


class UserDocumentsTextDataUpdateView(AdminAPIView):

    lookup_url_kwarg = 'user_id'

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_USERS)]

    arguments_serializer_class = UserDocumentsTextDataUpdateSerializer

    def get_queryset(self):
        return (
            User.objects
        )

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user = self.get_object()

        datasync = DataSyncDocumentsClient.from_settings()
        updater = UserProfileUpdater(user, datasync_client=datasync)

        updater.update_passport_textdata(dict(request.arguments['passport']))
        updater.update_driving_license_textdata(dict(request.arguments['driving_license']))

        return Response()


class UserDocumentPhotoBackgroundVideoUploaderView(AdminAPIView):

    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.VIEW_USER_DOCUMENT_VIDEOS),
    ]

    parser_classes = [
        ImageBinaryStreamParser,
        VideoBinaryStreamParser,
    ]

    mds_client = MDSDocumentsWrapper.from_settings()

    def do_put(self, request, user_id, document_id, photo_id):
        if not request.data:
            return Response(status=HTTP_400_BAD_REQUEST)

        document_id = parse_uuid_or_404(document_id)
        photo_id = parse_uuid_or_404(photo_id)
        document_photo = get_object_or_404(
            UserDocumentPhoto,
            user_id=user_id,
            document_id=document_id,
            id=photo_id,
        )

        updater = UserProfileUpdater(
            user=request.user,
            mds_client=self.mds_client,
        )
        updater.upload_document_photo_background_video(
            document_photo=document_photo,
            video_stream=request.data,
            mime_type=request.content_type,
        )

        return Response()


class UserDocumentPhotoBackgroundVideoConverterView(AdminAPIView):

    permission_classes = [
        AdminPermissionFactory.build(AdminPermissionCode.VIEW_USER_DOCUMENT_VIDEOS),
    ]

    mds_client = MDSDocumentsWrapper.from_settings()

    def do_post(self, request, user_id, document_id, photo_id):
        photo_id = parse_uuid_or_404(photo_id)
        document_photo = get_object_or_404(
            UserDocumentPhoto,
            user_id=user_id,
            id=photo_id,
        )

        updater = UserProfileUpdater(
            user=request.user,
            mds_client=self.mds_client,
        )
        updater.convert_document_photo_background_video(
            document_photo=document_photo,
        )

        return Response()
