from django.db.models import F, Func
from django.shortcuts import get_object_or_404
# from rest_framework.mixins import RetrieveModelMixin
from rest_framework.response import Response

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.users.core.user_profile_updater import UserProfileUpdater
from cars.users.models.user import User
from ..serializers.user import UserSerializer
from ..serializers.user_tags import UserTagsArgumentsSerializer
from .base import AdminAndCallcenterAPIView


class UserTagListView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    def do_get(self, request):  # pylint: disable=unused-argument
        tags = list(
            User.objects
            .annotate(
                all_tags=Func(F('tags'), function='unnest'),
            )
            .values_list('all_tags', flat=True)
            .distinct()
        )
        return Response(
            data={
                'tags': tags,
            },
        )


class UserTagsView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_all_general', 'support_py_sup_general', 'support_py_general'), require_all=False)]

    lookup_url_kwarg = 'user_id'
    queryset = User.objects.all()

    arguments_serializer_class = UserTagsArgumentsSerializer

    def do_post(self, request, user_id):  # pylint: disable=unused-argument
        user = get_object_or_404(User.objects.with_related(), id=user_id)
        UserProfileUpdater(user).add_tags(request.arguments['tags'], request.user.id)
        data = UserSerializer(user, context=self.get_serializer_context()).data
        return Response(data)

    def do_delete(self, request, user_id):
        user = get_object_or_404(User.objects.with_related(), id=user_id)
        UserProfileUpdater(user).remove_tags(request.arguments['tags'], request.user.id)
        data = UserSerializer(user, context=self.get_serializer_context()).data
        return Response(data)
