from rest_framework.response import Response
from rest_framework.status import HTTP_404_NOT_FOUND, HTTP_400_BAD_REQUEST

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.core.passport import PassportClient
from cars.registration_yang.core.assignment import AssignmentManager
from cars.registration_yang.models import YangAssignment
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.models import User
from ..serializers.yang_assignment import YangAssignmentVerboseSerializer


class UserIdByYangIdView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def do_get(self, *args, **kwargs):  # pylint:disable=unused-argument
        assignment = (
            YangAssignment.objects
            .filter(
                id=kwargs.get('secret_id'),
            )
            .select_related('license_back__user')
            .first()
        )

        if not assignment:
            return Response(status=HTTP_404_NOT_FOUND)

        return Response(
            {
                'user_id': assignment.license_back.user.id
            }
        )


class ResendUserForVerification(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._manager = AssignmentManager.from_settings()

    def do_post(self, *args, **kwargs):  # pylint: disable=unused-argument
        user_id = kwargs.get('user_id')

        try:
            self._manager.recreate_yang_assignment_for_user(user_id)
        except self._manager.AssignmentCreationError as e:
            # Unable to create complete assignment
            data = {
                'message': str(e),
            }
            return Response(data, status=HTTP_400_BAD_REQUEST)

        return Response()


class UserYangVerificationHistoryView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    serializer_class = YangAssignmentVerboseSerializer
    datasync = DataSyncDocumentsClient.from_settings(
        passport_client=PassportClient.from_settings_admin_api()
    )

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['datasync_client'] = self.datasync
        return context

    def get_queryset(self):
        return (
            YangAssignment.objects
            .filter(
                license_back__user=self.kwargs['user_id'],
            )
            .select_related(
                'license_back__user'
            )
        )

    def do_get(self, request, *args, **kwargs):
        if not self.check_banned_roles(request.user):
            return Response([])

        qs = self.get_queryset()

        user = User.objects.filter(id=self.kwargs['user_id']).first()
        if user is None or user.status == 'deleted':
            return Response([])

        return Response(
            [
                self.serializer_class(assignment, context=self.get_serializer_context()).data
                for assignment in qs
            ]
        )
