from django.utils.encoding import force_text
from rest_framework import filters
from rest_framework.compat import coreapi, coreschema


class CarsFilterMixin(object):

    operators_param = 'operators'
    operators_param_description = 'A list of operators for filtering'

    models_param = 'models'
    models_param_description = 'A list of models for filtering'

    min_fuel_param = 'min_fuel'
    min_fuel_param_description = 'Return cars with at least this fuel level'

    def get_schema_fields(self, _):
        fields = []
        fields.append(
            coreapi.Field(
                name=self.operators_param,
                required=False,
                location='query',
                schema=coreschema.Array(
                    items=coreschema.String(),
                    description=force_text(self.operators_param_description),
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.models_param,
                required=False,
                location='query',
                schema=coreschema.Array(
                    items=coreschema.String(),
                    description=force_text(self.models_param_description),
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.min_fuel_param,
                required=False,
                location='query',
                schema=coreschema.Number(
                    minimum=0,
                    maximum=100,
                    description=force_text(self.min_fuel_param_description),
                )
            )
        )
        return fields


class FreeCarsFilter(CarsFilterMixin, filters.BaseFilterBackend):

    limit_param = 'limit'
    limit_param_description = 'Max number of cars to return'

    sw_param = 'sw'
    sw_param_description = '"lon,lat" pair of southeastern point of the bounding rectangle'

    ne_param = 'ne'
    ne_param_description = '"lon,lat" pair of northwestern point of the bounding rectangle'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = []
        fields.append(
            coreapi.Field(
                name=self.limit_param,
                required=False,
                location='query',
                schema=coreschema.Integer(
                    description=force_text(self.limit_param_description),
                    default=True,
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.sw_param,
                required=False,
                location='query',
                schema=coreschema.String(
                    description=force_text(self.sw_param_description),
                    default=True,
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.ne_param,
                required=False,
                location='query',
                schema=coreschema.String(
                    description=force_text(self.ne_param_description),
                    default=True,
                )
            )
        )
        fields.extend(super().get_schema_fields(view))
        return fields


class ModifiedSinceFilter(filters.BaseFilterBackend):

    since_param = 'since'
    since_param_description = 'Interval start time as UTC unix timestamp'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = super().get_schema_fields(view)
        fields.append(
            coreapi.Field(
                name=self.since_param,
                required=True,
                location='query',
                schema=coreschema.Number(
                    description=force_text(self.since_param_description)
                )
            )
        )
        return fields


class NearestCarsFilter(CarsFilterMixin, filters.BaseFilterBackend):

    center_param = 'll'
    center_param_description = 'Central point "lon,lat"'

    limit_param = 'limit'
    limit_param_description = 'Max number of cars to return'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = []
        fields.append(
            coreapi.Field(
                name=self.center_param,
                required=True,
                location='query',
                schema=coreschema.String(
                    description=force_text(self.center_param_description)
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.limit_param,
                required=True,
                location='query',
                schema=coreschema.Integer(
                    description=force_text(self.limit_param_description),
                    default=True,
                )
            )
        )
        fields.extend(super().get_schema_fields(view))
        return fields


class CitiesFilter(filters.BaseFilterBackend):

    ll_param = 'll'
    ll_param_description = 'Location in "lon,lat" format'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = super().get_schema_fields(view)
        fields.append(
            coreapi.Field(
                name=self.ll_param,
                required=False,
                location='query',
                schema=coreschema.String(
                    description=force_text(self.ll_param_description)
                )
            )
        )
        return fields


class OperatorListFilter(filters.BaseFilterBackend):

    area_param = 'area'
    area_param_description = 'Set to "false" to exclude area polygon from the response'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = super().get_schema_fields(view)
        fields.append(
            coreapi.Field(
                name=self.area_param,
                required=False,
                location='query',
                schema=coreschema.Boolean(
                    description=force_text(self.area_param_description),
                    default=True,
                )
            )
        )
        return fields


class WalkAreasFilter(filters.BaseFilterBackend):

    ll_param = 'll'
    ll_param_description = 'User position in "lon,lat" format'

    min_walk_time_param = 'min_walk_time'
    min_walk_time_param_description = 'Minimal desired walk time in seconds'

    max_walk_time_param = 'max_walk_time'
    max_walk_time_param_description = 'Maximal desired walk time in seconds'

    num_areas_param = 'num_areas'
    num_areas_param_description = 'Number of areas to return'

    def filter_queryset(self, request, queryset, view):
        return queryset

    def get_schema_fields(self, view):
        fields = super().get_schema_fields(view)
        fields.append(
            coreapi.Field(
                name=self.ll_param,
                required=True,
                location='query',
                schema=coreschema.String(
                    description=force_text(self.ll_param_description)
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.min_walk_time_param,
                required=True,
                location='query',
                schema=coreschema.Number(
                    description=force_text(self.min_walk_time_param_description)
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.max_walk_time_param,
                required=True,
                location='query',
                schema=coreschema.Number(
                    description=force_text(self.max_walk_time_param_description)
                )
            )
        )
        fields.append(
            coreapi.Field(
                name=self.num_areas_param,
                required=True,
                location='query',
                schema=coreschema.Integer(
                    description=force_text(self.num_areas_param_description)
                )
            )
        )
        return fields
