from django.utils import translation
from rest_framework import serializers

import cars.settings
from .common import validate_point
from .car_model import CarModelSerializer
from .operator import LocalOperatorSerializer
from ..static_data import clients, operators


class BaseCarsArgumentsSerializer(serializers.Serializer):

    limit = serializers.IntegerField(min_value=0, required=False)


class FreeCarsArgumentsSerializer(BaseCarsArgumentsSerializer):

    sw = serializers.CharField(required=False)  # pylint: disable=invalid-name
    ne = serializers.CharField(required=False)  # pylint: disable=invalid-name

    def validate(self, data):
        any_coordinate_present = 'sw' in data or 'ne' in data
        both_coordinates_present = 'sw' in data and 'ne' in data
        if any_coordinate_present and not both_coordinates_present:
            raise serializers.ValidationError('Both sw and ne must be provided')

        if both_coordinates_present:
            data['rect'] = (data['sw'], data['ne'])

        return data

    def validate_sw(self, value):
        return validate_point(value)

    def validate_ne(self, value):
        return validate_point(value)


class NearestCarsArgumentsSerializer(BaseCarsArgumentsSerializer):

    ll = serializers.CharField()  # pylint: disable=invalid-name
    limit = serializers.IntegerField(min_value=0, max_value=15)

    def validate_ll(self, value):
        return validate_point(value)


class CarSerializer(serializers.Serializer):

    id = serializers.CharField()
    local_id = serializers.CharField()
    color = serializers.CharField()
    fuel = serializers.FloatField()
    is_free = serializers.BooleanField()
    lat = serializers.FloatField()
    lon = serializers.FloatField()
    operator = serializers.CharField()
    model = serializers.CharField()
    plate_number = serializers.CharField()

    address = serializers.SerializerMethodField()
    deeplink = serializers.SerializerMethodField()
    tariff = serializers.SerializerMethodField()
    walk_time = serializers.ReadOnlyField()

    def get_address(self, obj):
        return obj.localized_address(language=translation.get_language())

    def get_deeplink(self, obj):
        scheme_only = False

        # Hack until Transport doesn't support cyrillic deeplinks.
        request = self.context.get('request')
        if request and cars.settings.FEATURES['scheme_only_delimobil_for_transport']:
            if (request.client
                    and request.client == clients.TRANSPORT
                    and obj.operator_obj.short_name == operators.DELIMOBIL.short_name):
                scheme_only = True

        return obj.get_deeplink(scheme_only=scheme_only)

    def get_tariff(self, obj):
        return {
            'discount': obj.discount,
            'usage': obj.usage_tariff * (1 - obj.discount),
            'parking': obj.parking_tariff * (1 - obj.discount),
        }


class CarDetailsSerializer(CarSerializer):
    photo = serializers.URLField()
    operator_obj = LocalOperatorSerializer()
    model_obj = CarModelSerializer()


class TimestampedCarsSerializer(serializers.Serializer):
    cars = CarSerializer(many=True)
    timestamp = serializers.FloatField()


class TimestampedCarDetailsSerializer(serializers.Serializer):
    cars = CarDetailsSerializer(many=True)
    timestamp = serializers.FloatField()
