import os

import furl
import shapely.geometry
from django.conf import settings
from django.utils import translation
from rest_framework import serializers

from cars.django.serializers import BaseSerializer
from .common import validate_point


class CitySerializer(BaseSerializer):
    id = serializers.IntegerField(source='code')
    name = serializers.SerializerMethodField()
    icon_url = serializers.SerializerMethodField()
    location = serializers.SerializerMethodField()
    polygon = serializers.SerializerMethodField()

    def get_name(self, obj):
        return obj.localized_name(translation)

    def get_location(self, obj):
        return shapely.geometry.mapping(obj.location)

    def get_polygon(self, obj):
        return shapely.geometry.mapping(obj.polygon)

    def get_icon_url(self, obj):
        filename = '{}.png'.format(obj.short_name)
        path = os.path.join(settings.STATIC_URL, 'city-icons', filename)
        args = {
            'rev': 7,
        }
        request = self.context['request']
        img_url = request.build_absolute_uri(path)
        img_url = furl.furl(img_url).set(args=args).url
        return img_url


class CitiesWithCurrentSerializer(BaseSerializer):
    cities = CitySerializer(many=True)
    current_city_id = serializers.IntegerField()


class CitiesArgumentsSerializer(BaseSerializer):
    ll = serializers.CharField(required=False)  #  pylint: disable=invalid-name

    def validate_ll(self, value):
        return validate_point(value)
