from rest_framework import serializers

from ..static_data import car_models, operators


class CarFilterSerializer(serializers.Serializer):
    models = serializers.ListField(
        child=serializers.CharField(),
        required=False,
    )
    exclude_models = serializers.ListField(
        child=serializers.CharField(),
        required=False,
    )
    operators = serializers.ListField(
        child=serializers.CharField(),
        required=False,
    )
    exclude_operators = serializers.ListField(
        child=serializers.CharField(),
        required=False,
    )
    min_fuel = serializers.FloatField(min_value=0, max_value=100, required=False)

    def to_internal_value(self, data):
        '''Allow both comma-separated string and a list for operators and models'''

        for field in ['models', 'exclude_models', 'operators', 'exclude_operators']:
            if field in data:
                data[field] = self._ensure_list(data.get(field))

        validated_data = super().to_internal_value(data)

        return validated_data

    def _ensure_list(self, value):
        if isinstance(value, str):
            value = value.split(',')
        return value

    def validate_models(self, value):
        models = set()
        for model_str in value:
            for model in model_str.split(','):
                if model not in car_models.CAR_MODELS:
                    raise serializers.ValidationError('Unknown car model: {}'.format(model))
                models.add(model)
        return models

    def validate_exclude_models(self, value):
        return self.validate_models(value)

    def validate_operators(self, value):
        # Each operator in the list is allowed to be either a single short_name
        # or a comma-separated list of short names for GET and POST requests compatibility.
        ops = set()
        for operator_str in value:
            for operator in operator_str.split(','):
                if operator not in operators.OPERATORS:
                    raise serializers.ValidationError('Unknown operator: {}'.format(operator))
                ops.add(operator)
        return ops

    def validate_exclude_operators(self, value):
        return self.validate_operators(value)
