import datetime
import os

import furl
from django.conf import settings
from django.utils import translation
from rest_framework import serializers

from cars.django.serializers import BaseSerializer
from .currency import CurrencySerializer


class OperatorColorSerializer(BaseSerializer):
    red = serializers.FloatField()
    green = serializers.FloatField()
    blue = serializers.FloatField()


class OperatorIconsSerializer(BaseSerializer):
    map = serializers.URLField(source='map_default')
    map_selected = serializers.URLField()
    map_zoom = serializers.URLField()

    logo = serializers.SerializerMethodField()

    def get_logo(self, _):
        args = {
            'rev': 3,
        }
        operator = self.context['operator'].short_name
        path = '/{}/static/operator-icons/{}.png'.format(settings.API['location_prefix'], operator)
        request = self.context['request']
        url = request.build_absolute_uri(path)
        url = furl.furl(url).add(args=args).url
        return url


class SimpleOperatorSerializer(BaseSerializer):
    short_name = serializers.CharField()
    name = serializers.SerializerMethodField()

    def get_name(self, obj):
        return obj.localized_name(translation)


class OperatorSerializer(SimpleOperatorSerializer):
    color = OperatorColorSerializer()

    website = serializers.CharField()
    faq_url = serializers.CharField()

    ios_store_url = serializers.CharField()
    android_store_url = serializers.CharField()
    store_url = serializers.SerializerMethodField()

    icons = serializers.SerializerMethodField()

    def get_store_url(self, obj):
        return obj.ios_store_url

    def get_icons(self, obj):
        self.context['operator'] = obj
        icons_serializer = OperatorIconsSerializer(obj.icons, context=self.context)
        return icons_serializer.data


class LocalOperatorSerializer(OperatorSerializer):
    currency = CurrencySerializer()
    phone = serializers.CharField()
    info = serializers.SerializerMethodField()

    area = serializers.SerializerMethodField()
    area_img = serializers.SerializerMethodField()

    def get_info(self, obj):
        return obj.localized_info(translation)

    def get_area(self, obj):
        return obj.parkings.get_area(date_at=datetime.datetime.now(), format_='geojson')

    def get_area_img(self, obj):
        filename = '{}_{}.png'.format(obj.short_name, obj.city.short_name)
        path = os.path.join(settings.STATIC_URL, 'areas', filename)
        args = {
            'rev': 6,
        }
        request = self.context['request']
        img_url = request.build_absolute_uri(path)
        img_url = furl.furl(img_url).set(args=args).url
        return img_url

    def to_representation(self, instance):
        data = super().to_representation(instance)
        if not self.context['include_area']:
            data.pop('area')
        return data
