import cars.settings
from cars.core.filter import CarFilter
from .serializers import CarFilterSerializer
from .static_data import clients, operators


def available_cities_from_request(request):
    available_operators = available_operators_from_request(request)
    available_cities = {op.city for ops in available_operators.values() for op in ops}
    return available_cities


def available_operators_from_request(request):
    banned_for_client = cars.settings.FEATURES['banned_operators'].get(request.client, set())
    available_local_operators = {city: [op for op in ops.values() if op not in banned_for_client]
                                 for city, ops in operators.LOCAL_OPERATORS.items()}
    return available_local_operators


def city_models_from_request(request):
    available_operators = city_operators_from_request(request)
    banned_for_client = cars.settings.FEATURES['banned_models'].get(request.client, set())
    available_models = {m
                        for op in available_operators
                        for m in op.car_models
                        if m not in banned_for_client}
    return available_models


def city_operators_from_request(request):
    city_code = request.city.code if request.city else None
    banned_for_client = cars.settings.FEATURES['banned_operators'].get(request.client, set())
    local_operators = operators.LOCAL_OPERATORS.get(city_code, {})
    available_local_operators = [op
                                 for op in local_operators.values()
                                 if op not in banned_for_client]
    return available_local_operators


class CarFilterConstructionError(Exception):
    def __init__(self, errors, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.errors = errors


def car_filter_from_request(request):
    if request.method == 'GET':
        data = request.query_params.dict()
    else:
        data = request.data.get('filters', {})

    params = CarFilterSerializer(data=data)
    if not params.is_valid():
        raise CarFilterConstructionError(errors=params.errors)

    clean = params.validated_data

    filter_operators = clean.get('operators')
    if filter_operators is None:
        filter_operators = {op.short_name for op in city_operators_from_request(request)}
        exclude_operators = clean.get('exclude_operators')
        if exclude_operators:
            filter_operators -= exclude_operators

    filter_models = clean.get('models')
    if filter_models is None:
        filter_models = {m.short_name for m in city_models_from_request(request)}
        exclude_models = clean.get('exclude_models')
        if exclude_models:
            filter_models -= exclude_models

    car_filter = CarFilter(
        city_id=request.city.code if request.city else None,
        operators=filter_operators,
        models=filter_models,
        min_fuel=clean.get('min_fuel'),
    )

    return car_filter


def client_from_code(code):
    client = cars.settings.FEATURES['special_clients'].get(code, clients.AGGREGATOR)
    return client
