import shapely.geometry
from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import APIView

from cars.core.alerts import Alert, get_alerts_backend, NoAlertError
from cars.core.pedestrian_router import PedestrianRouter

from ..serializers import AlertCancellationSerializer, AlertSubscriptionSerializer
from ..util import CarFilterConstructionError, car_filter_from_request


class AlertCancellationView(APIView):

    def get_serializer(self, data=None):
        return AlertCancellationSerializer(data=data)

    def post(self, request, _=None):
        '''Cancel push notification subscription'''

        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            data = {
                'errors': serializer.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        backend = get_alerts_backend()
        try:
            backend.cancel_alert(platform=serializer.data['platform'],
                                 device_id=serializer.data['device_id'])
        except NoAlertError:
            data = {
                'errors': 'Alert not found',
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        return Response(status=status.HTTP_204_NO_CONTENT)


class AlertSubscriptionView(APIView):

    _prouter = PedestrianRouter()

    def get_serializer(self, data=None):
        return AlertSubscriptionSerializer(
            data=data,
            context={
                'request': self.request,
            },
        )

    def post(self, request, _=None):
        '''
        Subscribe to a push notification when a free car becomes available

        Parameters:
          * `platform`: Device plaftorm: ios/android
          * `device_id`: User device id to use as push notification's target
          * `location`: User location as a GeoJSON Point
          * `walk_time`: Max desired walk time
          * `filters`: A set of filters:
              * `operators`: A list of operator short names
              * `models`: A list of car models short names
              * `min_fuel`: Minimal fuel level

        Sample request:
        ```
        {
            "platform": "ios",
            "device_id": 42,
            "location": {
                "type": "Point",
                "coordinates": [37.588998, 55.733916]
            },
            "walk_time": 300,
            "filters": {
                "operators": ["delimobil", "belka"],
                "min_fuel": 0.5
            }
        }
        ```
        '''

        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            data = {
                'errors': serializer.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        location = serializer.validated_data['location']['coordinates']
        walk_time = serializer.validated_data['walk_time']
        area = self._prouter.build_walk_area_from_time(
            point=location, walk_seconds=walk_time,
        )

        try:
            car_filter = car_filter_from_request(request)
        except CarFilterConstructionError as e:
            return Response(
                {
                    'errors': e.errors,
                },
                status=status.HTTP_400_BAD_REQUEST,
            )

        alert = Alert(
            platform=serializer.validated_data['platform'],
            device_id=serializer.validated_data['device_id'],
            location=shapely.geometry.Point(location),
            area=area,
            filter_=car_filter,
        )

        backend = get_alerts_backend()
        backend.upsert_alert(alert)

        return Response(status=status.HTTP_201_CREATED)
