# pylint: disable=trailing-whitespace
import datetime
import logging
import time

from django.conf import settings
from django.core.cache import cache
from rest_framework import status
from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from rest_framework.views import APIView

import cars.settings
from cars.core.saas_index import SaasIndex, SaasIndexCache
from cars.core.util import get_city

from ..filters import FreeCarsFilter, ModifiedSinceFilter, NearestCarsFilter
from ..serializers import (
    CarDetailsSerializer, FreeCarsArgumentsSerializer,
    NearestCarsArgumentsSerializer, TimestampedCarsSerializer, TimestampedCarDetailsSerializer,
)
from ..util import CarFilterConstructionError, car_filter_from_request


LOGGER = logging.getLogger(__name__)


def get_saas_index():
    config = settings.CAR_INDEX
    cache_backend = SaasIndexCache(backend=cache, timeout=config['CACHE_TIMEOUT'])
    index = SaasIndex(config['SAAS_CONFIG'], cache=cache_backend)
    return index

SAAS_INDEX = get_saas_index()


class CarDetailsView(APIView):
    '''
    Detailed information about a single car.

    The response contains all the fields as in
    free and nearest cars endpoints with some extra data.
    '''

    def get(self, request, car_id, _=None):  # pylint: disable=unused-argument
        car = SAAS_INDEX.get_car(car_id=car_id)
        if car is None:
            raise NotFound()

        context = {
            'request': request,
            'include_area': False,
        }
        data = CarDetailsSerializer(car, context=context).data

        return Response(data)


class FreeCarsView(APIView):
    '''
    All free cars

    This method should be invoked at the start of a session to populate the initial state.  
    Only free cars are returned.

    The response varies on `City: <geo_id>` header.
    '''

    filter_backends = [FreeCarsFilter]

    def get(self, request, _=None):
        params = FreeCarsArgumentsSerializer(data=request.query_params)
        if not params.is_valid():
            data = {
                'errors': params.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        try:
            car_filter = car_filter_from_request(request)
        except CarFilterConstructionError as e:
            return Response(
                {
                    'errors': e.errors,
                },
                status=status.HTTP_400_BAD_REQUEST,
            )

        free_cars = SAAS_INDEX.get_free_cars(
            car_filter=car_filter,
            limit=params.validated_data.get('limit'),
            rect=params.validated_data.get('rect'),
        )

        context = {
            'request': request,
        }
        serializer = TimestampedCarsSerializer(free_cars, context=context)

        return Response(serializer.data)


class ModifiedCarsView(APIView):

    filter_backends = [ModifiedSinceFilter]

    def get(self, request, _=None):
        '''
        Recently modified cars

        This method should be invoked periodically to update the state.  
        Modification of `is_free` attribute of a car tells that it became free or rented.  
        Other attributes may change aswell, e.g. position and fuel level.
        '''

        try:
            since_ts = float(request.query_params.get('since'))
            if time.time() - since_ts > cars.settings.CAR_INDEX['MAX_DELTA']:
                raise RuntimeError
            since = datetime.datetime.utcfromtimestamp(since_ts)
        except Exception:
            return Response({'error': 'Invalid "since" parameter'},
                            status=status.HTTP_400_BAD_REQUEST)

        try:
            car_filter = car_filter_from_request(request)
        except CarFilterConstructionError as e:
            return Response(
                {
                    'errors': e.errors,
                },
                status=status.HTTP_400_BAD_REQUEST,
            )
        car_filter.is_free = None  # Don't select only free cars which is the default.

        modified_cars = SAAS_INDEX.get_modified_cars(since, car_filter=car_filter)
        context = {
            'request': request,
        }
        serializer = TimestampedCarsSerializer(modified_cars, context=context)

        return Response(serializer.data)


class NearestCarsView(APIView):

    filter_backends = [NearestCarsFilter]

    def get(self, request, _=None):  # pylint: disable=unused-argument
        '''
        Nearest free cars
        '''

        params = NearestCarsArgumentsSerializer(data=request.query_params)
        if not params.is_valid():
            data = {
                'errors': params.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        # Determine city by coordinates for nearest cars view.
        center = params.validated_data['ll']
        center_lon, center_lat = center
        request.city = get_city(lat=center_lat, lon=center_lon)

        try:
            car_filter = car_filter_from_request(request)
        except CarFilterConstructionError as e:
            return Response(
                {
                    'errors': e.errors,
                },
                status=status.HTTP_400_BAD_REQUEST,
            )

        free_cars = SAAS_INDEX.get_nearest_cars(
            center=center,
            car_filter=car_filter,
            limit=params.validated_data.get('limit'),
            rect=params.validated_data.get('rect'),
        )
        context = {
            'include_area': False,
            'request': request,
        }
        serializer = TimestampedCarDetailsSerializer(free_cars, context=context)

        return Response(serializer.data)
