# pylint: disable=trailing-whitespace
import collections
import operator

import shapely.geometry
from rest_framework import status
from rest_framework.generics import ListAPIView
from rest_framework.response import Response
from rest_framework.views import APIView

from ..filters import CitiesFilter, OperatorListFilter
from ..serializers import (
    CarModelSerializer,
    ClientSerializer,
    CitiesArgumentsSerializer, CitiesWithCurrentSerializer,
    LocalOperatorSerializer,
)
from ..util import (
    available_cities_from_request,
    city_models_from_request, city_operators_from_request, client_from_code,
)


class GoneView(APIView):
    def get(self, request, *args, **kwargs):
        return Response(status=status.HTTP_410_GONE)

    def post(self, request, *args, **kwargs):
        return Response(status=status.HTTP_410_GONE)

    def put(self, request, *args, **kwargs):
        return Response(status=status.HTTP_410_GONE)

    def patch(self, request, *args, **kwargs):
        return Response(status=status.HTTP_410_GONE)

    def delete(self, request, *args, **kwargs):
        return Response(status=status.HTTP_410_GONE)


class ParametrizedAPIView(APIView):

    parameters_serializer_class = None

    def get_parameters(self, data):
        serializer = self.parameters_serializer_class(data=data)  # pylint: disable=not-callable
        serializer.is_valid(raise_exception=True)
        return serializer.data


class CarModelList(ListAPIView):
    '''
    Static car model data

    `short_name` serves as model's identifier.
    '''

    serializer_class = CarModelSerializer

    def get_queryset(self):
        models = city_models_from_request(self.request)
        return sorted(models, key=operator.attrgetter('id'))


class CitiesView(APIView):
    '''
    User city and a list of all available cities.
    '''

    CitiesWithCurrent = collections.namedtuple('CitiesWithCurrent', ['cities', 'current_city_id'])

    filter_backends = [CitiesFilter]

    def get(self, request, _=None):
        params = CitiesArgumentsSerializer(data=request.query_params)
        if not params.is_valid():
            data = {
                'errors': params.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        available_cities = available_cities_from_request(request)
        lon, lat = params.validated_data.get('ll', (None, None))
        current_city_id = self.get_current_city_id(lat=lat, lon=lon, choices=available_cities)
        obj = self.CitiesWithCurrent(cities=available_cities, current_city_id=current_city_id)
        context = {
            'request': request,
        }
        data = CitiesWithCurrentSerializer(obj, context=context).data

        return Response(data=data)

    def get_current_city_id(self, lat, lon, choices):
        if not (lat and lon):
            return None

        point = shapely.geometry.Point(lon, lat)

        for city in choices:
            if city.polygon.contains(point):
                return city.code

        return None


class OperatorsList(ListAPIView):
    '''
    Static operator data

    `short_name` serves as operator's identifier.
    '''

    filter_backends = [OperatorListFilter]
    serializer_class = LocalOperatorSerializer

    def get_queryset(self):
        available_operators = city_operators_from_request(self.request)
        return sorted(available_operators, key=operator.attrgetter('id'))

    def get_serializer_context(self):
        context = super().get_serializer_context()
        include_area = self.request.query_params.get('area', '').lower() != 'false'
        context['include_area'] = include_area
        return context


class ClientIdView(APIView):

    def get(self, request, _=None):
        code = request.query_params.get('code')
        client = client_from_code(code)
        data = ClientSerializer(client).data
        return Response(data)
