# coding: utf-8
from django.conf import settings
from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import APIView

import kubiki

import cars.settings
from cars.aggregator.core.push_client import AggregatorPushClientLogger
from cars.django.lock import UwsgiLock
from ..static_data.operators import LOCAL_OPERATORS


class LandingView(APIView):

    _push_client = AggregatorPushClientLogger(
        filename=settings.PUSH_CLIENT['landing']['filename'],
        lock=UwsgiLock(),
    )

    def post(self, request, _=None):
        headers = {k: v for k, v in request.META.items() if isinstance(v, str)}
        data = {
            'form': request.data,
            'headers': headers,
        }
        self._push_client.log(operator='', type_='landing', data=data)
        return Response()


class LandingSmsView(APIView):

    _text = 'Ссылка на приложение Яндекс.Каршеринг https://appsto.re/ru/4Cjjib.i'
    _yasms = kubiki.yasms.YaSms(
        url=cars.settings.YASMS['url'],
        sender=cars.settings.YASMS['sender'],
    )
    _push_client = AggregatorPushClientLogger(
        filename=settings.PUSH_CLIENT['landing']['filename'],
        lock=UwsgiLock(),
    )

    # Allow SMS request only to the countries of presence.
    _allowed_country_codes = {
        op.phone.split('(')[0].strip(' +')
        for operators_per_city in LOCAL_OPERATORS.values()
        for op in operators_per_city.values()
    }

    def post(self, request, _=None):
        phone = request.data.get('phone')
        if not phone:
            data = {
                'errors': {
                    'phone': 'Parameter is missing',
                },
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        normalized_phone = self._yasms.normalize_phone(phone)
        if not self._check_country_code(normalized_phone):
            data = {
                'errors': {
                    'phone': 'Unsupported country code',
                },
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        try:
            message_id = self._yasms.sendsms(phone=normalized_phone, text=self._text)
            error = None
        except Exception as e:
            message_id = None
            error = str(e)

        self._push_client.log(
            operator='',
            type_='landing.sms',
            data={
                'phone': phone,
                'message_id': message_id,
                'error': error,
            },
        )

        return Response()

    def _check_country_code(self, normalized_phone):
        if normalized_phone.startswith('8'):
            normalized_phone = normalized_phone[1:]

        for country_code in self._allowed_country_codes:
            if normalized_phone.startswith(country_code):
                return True

        return False
