import logging

import shapely.geometry
from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import APIView

import saaspy.client


LOGGER = logging.getLogger(__name__)


class LandmarksView(APIView):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.proxy = LandmarksProxy()

    def get(self, request):
        try:
            lon_str, lat_str = request.query_params['ll'].split(',')
            lon, lat = float(lon_str), float(lat_str)
            radius = int(request.query_params['radius'])
        except Exception as e:
            data = {
                'error': '{}: {}'.format(type(e).__name__, str(e)),
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        landmarks = self.proxy.get_landmarks(center=(lon, lat), radius=radius)
        data = {
            'landmarks': landmarks,
        }

        return Response(data)


class LandmarksProxy(object):

    def __init__(self):
        self.client = saaspy.client.GeoSaasClient(
            search_host='saas-searchproxy-maps-prestable.yandex.net:17000',
            service_name='pivotal_geo_objects',
            index_host=None,
            key=None,
        )

    def get_landmarks(self, center, radius):
        query = self.client.construct_complex_query({
            'type': 'rect',
            'rect': '{} {}'.format(center[0], center[1]),
            'd': radius,
        })
        response = self.client.search_by_text(
            query,
            extra_params={
                'component': 'GeoSearch',
            },
        )
        if response.status_code == 404:
            return []
        response.raise_for_status()
        landmarks = self._get_landmarks_from_response(response)
        return landmarks

    def _get_landmarks_from_response(self, response):
        landmarks = []
        for doc in response.documents:
            try:
                landmark = self._get_landmark_from_document(doc)
            except Exception:
                LOGGER.exception('Failed to parse landmark document')
                continue
            landmarks.append(landmark)
        return landmarks

    def _get_landmark_from_document(self, doc):
        lon_str, lat_str = doc['geoPolyline'].split(' ')
        lon, lat = float(lon_str), float(lat_str)
        point = shapely.geometry.Point(lon, lat)

        popularity = doc.properties.get('popularity')
        if popularity is not None:
            popularity = float(popularity)

        rate = doc.properties.get('rate')
        if rate is not None:
            rate = int(rate)

        return {
            'name': doc.properties['Name'],
            'address': doc.properties['address'],
            'popularity': popularity,
            'rate': rate,
            'location': shapely.geometry.mapping(point),
        }
