import shapely.geometry
from rest_framework import status
from rest_framework.response import Response
from rest_framework.views import APIView

import cars.settings
from cars.core.pusher import Pusher
from cars.core.saas_index import SaasIndex
from ..serializers import PushSerializer


class PushView(APIView):

    pusher = Pusher()
    saas_index = SaasIndex(cars.settings.CAR_INDEX['SAAS_CONFIG'])

    def get_serializer(self, data=None):
        return PushSerializer(data=data)

    def post(self, request, _=None):
        '''
        Sample request:
        ```
        {
            "device_id": "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
            "car_id": "delimobil-С021НХ777_МСК"
        }
        ```
        '''
        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            data = {
                'errors': serializer.errors,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        device_id = serializer.validated_data['device_id']

        car_id = serializer.validated_data['car_id']
        car = self.saas_index.get_car(car_id)
        if not car:
            return Response(data={
                'error': 'Car not found',
            }, status=status.HTTP_404_NOT_FOUND)

        location = shapely.geometry.Point([37.600480, 55.740792])
        self.pusher.send_free_car_push(platform='ios', token=device_id, location=location, car=car)

        return Response()
