import datetime
import logging

import dateutil.parser
from django.db import transaction
from django.db.models import F
from django.utils import timezone

from cars.core.util import datetime_helper
from cars.registration.core.chat.manager import ChatManager
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.core.user_profile_updater import UserProfileUpdater
from cars.users.models.user import User
from cars.users.models.user_documents import UserDocumentPhoto
from ..models.ban import Ban


LOGGER = logging.getLogger(__name__)


class BlockHandler:

    class Error(Exception):
        pass

    def __init__(self, chat_manager, datasync_client):
        self._chat_manager = chat_manager
        self._datasync = datasync_client

    def block(self, ban):
        chat_session = self.get_chat_session(user=ban.user)
        with transaction.atomic():
            chat_session.clear_chat(force=True)
            self._do_block(chat_session=chat_session)

    def unblock(self, user, operator, ban):
        if ban is not None:
            ban.finished_by = operator
            ban.finished_at = timezone.now()
            ban.save()

        try:
            UserProfileUpdater(user).update_status(User.Status.ACTIVE)
        except UserProfileUpdater.StatusChangeError:
            LOGGER.exception('status.invalid for user_id {}'.format(user.id))

    def _do_block(self, chat_session):
        raise NotImplementedError

    def try_unblock_all(self):
        pass

    def get_chat_session(self, user):
        chat_context = {
            'user': user,
        }
        chat_session = self._chat_manager.make_session(user=user, context=chat_context)
        return chat_session


class NotificationBlockHandlerFactory:

    @classmethod
    def build(cls, message_group_id):

        class NotificationBlockHandler(BlockHandler):
            def _do_block(self, chat_session):
                chat_session.send_message_group(id_=message_group_id)

        return NotificationBlockHandler


class OldLicenseBlockHandler(BlockHandler):

    def _do_block(self, chat_session):
        chat_session.send_message_group(id_='block_old_license_intro')
        chat_session.enqueue_chat_action(chat_action_id='block_old_license_1')
        chat_session.enqueue_chat_action(chat_action_id='block_old_license_2')

    def try_unblock_all(self):
        verified_photos = (
            UserDocumentPhoto.objects
            .select_related('user')
            .filter(
                type=UserDocumentPhoto.Type.DRIVER_LICENSE_BACK.value,
                verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                verified_at__gt=F('user__bans__started_at'),
                user__bans__reason=Ban.Reason.OLD_LICENSE.value,
                user__bans__finished_at__isnull=True,
            )
            .distinct()
        )
        candidate_users = {p.user for p in verified_photos}
        candidate_bans = (
            Ban.objects
            .select_related('user')
            .filter(
                user__in=candidate_users,
                reason=Ban.Reason.OLD_LICENSE.value,
                finished_at__isnull=True,
            )
        )

        for ban in candidate_bans:
            self.try_unblock(ban)

    def try_unblock(self, ban):
        verified_license_front = (
            UserDocumentPhoto.objects
            .filter(
                type=UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT.value,
                verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                verified_at__gt=ban.started_at,
            )
            .first()
        )
        verified_license_back = (
            UserDocumentPhoto.objects
            .filter(
                type=UserDocumentPhoto.Type.DRIVER_LICENSE_BACK.value,
                verification_status=UserDocumentPhoto.VerificationStatus.OK.value,
                verified_at__gt=ban.started_at,
            )
            .first()
        )
        if not verified_license_front or not verified_license_back:
            return

        if self._is_still_old_license(user=ban.user):
            return

        chat_session = self.get_chat_session(user=ban.user)
        with transaction.atomic():
            chat_session.send_message_group(id_='block_old_license_outro')
            self.unblock(
                user=ban.user,
                operator=None,
                ban=ban,
            )

    def _is_still_old_license(self, user):
        license_data = self._datasync.get_license_unverified(user.uid, user.driving_license_ds_revision)

        try:
            license_valid_to = self._get_license_end_date(license_data)
        except Exception:
            LOGGER.exception(
                'ban manager got bad format of license data for user {}: {}'
                .format(str(user.id), license_data)
            )
            return True

        is_still_old_license = license_valid_to <= timezone.now().date()
        return is_still_old_license

    def _get_license_end_date(self, license_data):
        _, license_valid_to = self._get_category_b_span(license_data)

        if not license_valid_to:
            raise Exception('cannot obtain license end date')

        return license_valid_to

    def _get_category_b_span(self, license_data):
        # copycat of method from registration manager; no reason to refactor now
        if 'B' not in license_data['categories'].upper():
            return None, None

        if 'experience_from' in license_data:
            start_date_str = license_data['experience_from']
        elif 'categories_b_valid_from_date' in license_data:
            start_date_str = license_data['categories_b_valid_from_date']
        else:
            issue_date = dateutil.parser.parse(license_data['issue_date']).date()

            # Old format driver license.
            if issue_date > datetime.date(2014, 1, 1):
                masked_license_data = {k: '***' for k in license_data}
                LOGGER.error('invalid new style driver license: %s', masked_license_data)
                start_date_str = None
            else:
                start_date_str = license_data['issue_date']

        if start_date_str is None:
            start_date = None
        else:
            start_date = dateutil.parser.parse(start_date_str).date()

        end_date_str = license_data.get('categories_b_valid_to_date')
        if end_date_str is not None:
            end_date = dateutil.parser.parse(end_date_str).date()
        else:
            issue_date = dateutil.parser.parse(license_data['issue_date']).date()
            end_date = datetime_helper.add_years(issue_date, years=10)

        # Infer start_date/end_date from each other if any one of them is missing.
        if start_date and not end_date:
            end_date = datetime_helper.add_years(start_date, years=10)
        if not start_date and end_date:
            start_date = datetime_helper.add_years(end_date, years=-10)

        return start_date, end_date


class BanManager:

    handlers = {
        Ban.Reason.AUTO: NotificationBlockHandlerFactory.build(
            'block_other',
        ),
        Ban.Reason.DRIVING_BAN: NotificationBlockHandlerFactory.build(
            'block_driving_ban',
        ),
        Ban.Reason.DUPLICATE_LICENSE: NotificationBlockHandlerFactory.build(
            'block_duplicate_driver_license',
        ),
        Ban.Reason.DUPLICATE_PASSPORT: NotificationBlockHandlerFactory.build(
            'block_duplicate_namber_pass',
        ),
        Ban.Reason.OLD_LICENSE: OldLicenseBlockHandler,
        Ban.Reason.SPEED_ASSHOLE: NotificationBlockHandlerFactory.build(
            'block_speed_asshole',
        ),
        Ban.Reason.TOO_OLD: NotificationBlockHandlerFactory.build(
            'block_other',
        ),
        Ban.Reason.OTHER: NotificationBlockHandlerFactory.build(
            'block_other',
        ),
    }

    class Error(Exception):
        pass

    def __init__(self, chat_manager, datasync_client):
        self._chat_manager = chat_manager
        self._datasync = datasync_client

    @classmethod
    def from_settings(cls):
        return cls(
            chat_manager=ChatManager.from_settings(),
            datasync_client=DataSyncDocumentsClient.from_settings(),
        )

    def initialize_handler(self, handler_class):
        return handler_class(
            chat_manager=self._chat_manager,
            datasync_client=self._datasync,
        )

    def block(self, user, operator, reason, comment=None):
        LOGGER.info('block requested for user %s from %s', user, operator)

        registration_state = user.get_registration_state()
#        if registration_state is None or registration_state.chat_completed_at is None:
#            raise self.Error('chat.incomplete')

        with transaction.atomic():
            user = User.objects.select_for_update().get(id=user.id)

            if user.get_status() is User.Status.BLOCKED:
                LOGGER.info('user %s is already blocked', user.id)
            else:
                try:
                    UserProfileUpdater(user).update_status(User.Status.BLOCKED)
                except UserProfileUpdater.StatusChangeError:
                    raise self.Error('status.invalid')

            if Ban.objects.filter(user=user, finished_at__isnull=True).exists():
                LOGGER.info('ban already exists for user %s', user.id)
                return

            ban = Ban.objects.create(
                user=user,
                started_by=operator,
                started_at=timezone.now(),
                reason=reason.value,
                comment=comment,
            )

            handler_class = self.handlers[reason]
            handler = self.initialize_handler(handler_class)
            handler.block(ban)

        LOGGER.info('block successfull for user %s from %s', user, operator)

        return user

    def unblock(self, user, operator):
        LOGGER.info('unblock requested for user %s from %s', user, operator)
        with transaction.atomic():
            user = User.objects.select_for_update().get(id=user.id)
            if user.get_status() is User.Status.ACTIVE:
                LOGGER.info('user %s is already active', user.id)
                return

            if user.get_status() is not User.Status.BLOCKED:
                raise self.Error('status.invalid')

            ban = (
                Ban.objects
                .filter(
                    user=user,
                    finished_at__isnull=True,
                )
                .order_by('-started_at')
                .first()
            )
            if ban is None:
                LOGGER.warning('missing ban for a blocked user %s', user.id)
                reason = Ban.Reason.OTHER
            else:
                reason = ban.get_reason()

            handler_class = self.handlers[reason]
            handler = self.initialize_handler(handler_class)
            try:
                handler.unblock(user=user, operator=operator, ban=ban)
            except handler.Error as e:
                raise self.Error(str(e))

        LOGGER.info('unblock successfull for user %s from %s', user, operator)

        return user

    def try_unblock_all(self):
        for handler_class in self.handlers.values():
            handler = self.initialize_handler(handler_class)
            handler.try_unblock_all()
