import decimal

from django.db import transaction
from django.utils import timezone

from ..models.bonus_account import BonusAccount, BonusAccountOperation


class BonusAccountManager:

    class Error(Exception):
        pass

    class InsufficientFundsError(Error):
        pass

    def __init__(self, account):
        self._account = account

    @property
    def account(self):
        return self._account

    @classmethod
    def from_user(cls, user):
        """Get manager instance for user's account creating one if it's missing"""

        account = user.get_bonus_account()
        if account is None:
            account = BonusAccount.objects.create(user=user)

        mgr = cls(account=account)

        return mgr

    def debit_generic(self, amount, comment, operator, nonce):
        return self._update_generic(
            amount=amount,
            comment=comment,
            operator=operator,
            nonce=nonce,
        )

    def credit_generic(self, amount, comment, operator, nonce):
        return self._update_generic(
            amount=-amount,
            comment=comment,
            operator=operator,
            nonce=nonce,
        )

    def _update_generic(self, amount, comment, operator, nonce):
        with transaction.atomic():
            self._update_and_lock_account()

            operation = BonusAccountOperation.objects.filter(nonce=nonce).first()
            if operation is not None:
                return self._account

            if amount < 0 and self._account.generic_balance < -amount:
                raise self.InsufficientFundsError

            self._account.generic_earned += amount

            self._sync_balance()
            self._account.save()

            BonusAccountOperation.objects.create(
                bonus_account=self._account,
                created_at=timezone.now(),
                created_by=operator,
                amount=amount,
                balance=self._account.balance,
                comment=comment,
                nonce=nonce,
            )

        return self._account

    def update_registration_taxi_cashback_earned(self, value):
        with transaction.atomic():
            self._update_and_lock_account()

            # Registration taxi cashback is not expected to decrease.
            assert value >= self._account.registration_taxi_cashback_earned

            self._account.registration_taxi_cashback_earned = value
            self._sync_balance()
            self._account.save()

        return self._account

    def withdraw(self, amount):
        with transaction.atomic():
            self._update_and_lock_account()

            if self._account.balance < amount:
                raise self.InsufficientFundsError

            amount_to_withdraw = amount
            if self._account.registration_taxi_cashback_balance > 0:
                withdraw_from_registration_taxi_cashback = min(
                    amount_to_withdraw,
                    self._account.registration_taxi_cashback_balance,
                )
                self._account.registration_taxi_cashback_spent += (
                    withdraw_from_registration_taxi_cashback
                )
                amount_to_withdraw -= withdraw_from_registration_taxi_cashback

            if self._account.generic_balance > 0:
                withdraw_from_generic = min(
                    amount_to_withdraw,
                    self._account.generic_balance,
                )
                self._account.generic_spent += withdraw_from_generic
                amount_to_withdraw -= withdraw_from_generic

            assert amount_to_withdraw == 0

            self._sync_balance()
            self._account.save()

        return self._account

    def _update_and_lock_account(self):
        self._account = BonusAccount.objects.select_for_update().get(id=self._account.id)

    def _sync_balance(self):
        self._account.balance = decimal.Decimal(0)
        self._account.balance += self._account.generic_balance
        self._account.balance += self._account.registration_taxi_cashback_balance
