import concurrent.futures
import datetime
import decimal
import logging
import time

import pytz
from cars.orders.models import OrderItem
from django.db import transaction
from django.db.models import Q
from django.utils import timezone

import cars.settings
from cars.core.trust import TrustClient
from ..core.bonus_account_manager import BonusAccountManager
from ..models.bonus_payment import BonusPayment
from ..models.card_payment import CardPayment


LOGGER = logging.getLogger(__name__)


class PaymentProcessor:

    def __init__(self, bonus_processor, card_processor):
        self._bonus_processor = bonus_processor
        self._card_processor = card_processor

    @classmethod
    def from_settings(cls, push_client):
        bonus_processor = BonusPaymentProcessor.from_settings(push_client=push_client)
        card_processor = CardPaymentProcessor.from_settings(push_client=push_client)
        return cls(
            bonus_processor=bonus_processor,
            card_processor=card_processor,
        )

    def try_make_bonus_payment(self, user, max_amount):
        return self._bonus_processor.try_make_payment(user=user, max_amount=max_amount)

    def make_card_payment(self, user, amount, paymethod_id):
        now = timezone.now()
        payment = CardPayment.objects.create(
            user=user,
            amount=amount,
            status=CardPayment.Status.DRAFT.value,
            created_at=now,
            updated_at=now,
            paymethod_id=paymethod_id,
        )
        return payment

    def process_all(self):
        self._card_processor.process_all()

    def process(self, payment, card_processor_kwargs=None):
        if card_processor_kwargs is None:
            card_processor_kwargs = {}

        if isinstance(payment, CardPayment):
            self._card_processor.process(payment=payment, **card_processor_kwargs)
        else:
            raise RuntimeError('unreachable: {}'.format(payment))

    def refund(self, payment):
        assert isinstance(payment, CardPayment)
        self._card_processor.refund_payment(payment)

    def resize(self, payment, amount):
        assert isinstance(payment, CardPayment)
        self._card_processor.resize_payment(payment=payment, amount=amount)

    def wait_for_completion(self, payments, timeout):
        """
        Wait for payment completion at most for timeout seconds.

        Return a list of processed payments with None at places of timeouted operations.
        """

        if not payments:
            return []

        payment = payments[0]
        if isinstance(payment, BonusPayment):
            processor = self._bonus_processor
        elif isinstance(payment, CardPayment):
            processor = self._card_processor
        else:
            raise RuntimeError('unreachable: {}'.format(payment))

        result = processor.wait_for_completion(payments=payments, timeout=timeout)

        return result


class BonusPaymentProcessor:

    @classmethod
    def from_settings(cls, push_client):  # pylint: disable=unused-argument
        return cls()

    def try_make_payment(self, user, max_amount):
        mgr = BonusAccountManager.from_user(user)

        # The loop handles concurrent withdraws.
        while True:
            if mgr.account.balance <= 0:
                payment = None
                break

            amount = min(mgr.account.balance, max_amount)

            try:
                with transaction.atomic():
                    mgr.withdraw(amount)
                    payment = BonusPayment.objects.create(
                        user=user,
                        amount=amount,
                        created_at=timezone.now(),
                    )
                break
            except BonusAccountManager.InsufficientFundsError:
                continue

        return payment

    def wait_for_completion(self, payments, timeout):  # pylint: disable=unused-argument
        for p in payments:
            assert isinstance(p, BonusPayment)
        return payments


class CardPaymentProcessor:

    def __init__(self, trust_client, fiscal_title, fiscal_nds, clear_delay):
        """
        Args:
          trust_client: TrustClient instance.
          fiscal_title: Product title in check.
          fiscal_nds: VAT amount in check.
          clear_delay: datetime.timedelta of time until payments stay authorized.
        """
        self._client = trust_client
        self._fiscal_title = fiscal_title  # deprecated
        self._fiscal_nds = fiscal_nds
        self._clear_delay = clear_delay

    @classmethod
    def from_settings(cls, push_client):
        settings = cars.settings.BILLING['cards']
        trust_client = TrustClient.from_settings(push_client=push_client)
        return cls(
            trust_client=trust_client,
            fiscal_title=settings['fiscal_title'],
            fiscal_nds=settings['fiscal_nds'],
            clear_delay=settings['clear_delay'],
        )

    def process_all(self):
        min_clearable_start_date = timezone.now() - self._clear_delay
        payments = (
            CardPayment.objects
            .select_related(
                'user'
            )
            .prefetch_related(
                'order_item_payments__order_item'
            )
            .exclude(
                Q(status__in=[x.value for x in CardPayment.FINAL_STATUSES])
                |
                Q(
                    Q(status=CardPayment.Status.AUTHORIZED.value)
                    &
                    Q(started_at__gt=min_clearable_start_date)
                )
            )
            .order_by('-started_at')
        )
        payments_count = payments.count()

        if payments_count == 0:
            return

        max_workers = min(
            64,
            max(
                1,
                payments_count / 4,
            ),
        )

        futures = []
        with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as tp:
            for payment in payments:
                future = tp.submit(self.process, payment)
                futures.append((payment, future))

        for payment, future in futures:
            try:
                future.result()
            except Exception:
                LOGGER.exception('failed to process card payment %s', payment.id)
                continue

    def process(self, payment, force_clear=False):
        LOGGER.info('processing card payment %s', payment.id)

        try:
            with transaction.atomic():
                payment = CardPayment.objects.select_for_update().get(id=payment.id)

                processed = False

                # Payments shouldn't be initialized and started in the same transaction.
                # Otherwise it is possible to start payment and rollback CardPayment creation.
                if payment.get_status() is CardPayment.Status.DRAFT:
                    processed = True
                    self.initialize_payment(payment)
                elif payment.get_status() is CardPayment.Status.NOT_STARTED:
                    processed = True
                    self.start_payment(payment)

                if payment.get_status() is CardPayment.Status.STARTED:
                    processed = True
                    self.update_started_payment(payment)
                if payment.get_status() is CardPayment.Status.AUTHORIZED:
                    processed = True
                    self.try_complete_authorized_payment(payment, force=force_clear)
                if payment.get_status() is CardPayment.Status.NOT_AUTHORIZED:
                    processed = True

                if not processed:
                    raise RuntimeError('unreachable: {}'.format(payment.get_status()))
        except self._client.BadRequestError:
            # Bad request errors may be caused by out-of-sync payment statuses.
            self.sync_payment(payment)
            raise

        return payment

    def initialize_payment(self, payment):
        if payment.get_status() is CardPayment.Status.NOT_STARTED:
            LOGGER.info(
                'payment %s with purchase_token %s is already initialized',
                payment.id,
                payment.purchase_token,
            )
            return

        assert payment.get_status() is CardPayment.Status.DRAFT

        fiscal_title = 'Услуга бронирования транспортного средства'
        for order_item_payment in payment.order_item_payments.all():
            if order_item_payment.order_item.type == OrderItem.Type.CARSHARING_RIDE.value:
                fiscal_title = 'Арендная плата за аренду транспортного средства'
                break

        try:
            create_response = self._client.create_payment(
                uid=payment.user.uid,
                paymethod_id=payment.paymethod_id,
                amount=payment.amount,
                user_email=payment.user.email,
                fiscal_title=fiscal_title,
                fiscal_nds=self._fiscal_nds,
            )
        except self._client.InvalidPaymentMethodError:
            payment.status = CardPayment.Status.INVALID_PAYMENT_METHOD.value
            payment.save()
            return

        payment.purchase_token = create_response['purchase_token']
        self.sync_payment(payment)

        if payment.get_status() is not CardPayment.Status.NOT_STARTED:
            raise RuntimeError(
                'unexpected card payment status for {}: {}'.format(
                    payment.id,
                    payment.get_status(),
                )
            )

        LOGGER.info(
            'payment %s with purchase_token %s is successfully initialized',
            payment.id,
            payment.purchase_token,
        )

        return payment

    def start_payment(self, payment):
        if payment.get_status() is CardPayment.Status.STARTED:
            LOGGER.info(
                'payment %s with purchase_token %s is already started',
                payment.id,
                payment.purchase_token,
            )
            return

        assert payment.get_status() is CardPayment.Status.NOT_STARTED

        response = self._client.start_payment(
            uid=payment.user.uid,
            purchase_token=payment.purchase_token,
        )

        self.sync_payment_from_response(payment=payment, response=response)

        expected_statuses = {
            CardPayment.Status.AUTHORIZED,
            CardPayment.Status.NOT_AUTHORIZED,
            CardPayment.Status.STARTED,
        }

        if payment.get_status() in expected_statuses:
            LOGGER.info(
                'payment %s with purchase_token %s is successfully started',
                payment.id,
                payment.purchase_token,
            )
        else:
            raise RuntimeError('unexpected payment status: %s', payment.get_status())

    def update_started_payment(self, payment):
        assert payment.get_status() is CardPayment.Status.STARTED
        self.sync_payment(payment)

    def try_complete_authorized_payment(self, payment, force=False):
        assert payment.get_status() is CardPayment.Status.AUTHORIZED

        if not force and timezone.now() - payment.started_at < self._clear_delay:
            return

        try:
            self._client.clear_payment(
                uid=payment.user.uid,
                purchase_token=payment.purchase_token,
            )
        except self._client.Error:
            LOGGER.exception('failed to clear payment: %s', payment.id)

        self.sync_payment(payment)

    def resize_payment(self, payment, amount, qty=0):
        with transaction.atomic():
            payment = CardPayment.objects.select_for_update().get(id=payment.id)
            assert payment.get_status() is CardPayment.Status.AUTHORIZED

            self._client.resize_payment(
                uid=payment.user.uid,
                purchase_token=payment.purchase_token,
                amount=amount,
                qty=qty,
            )
            self.sync_payment(payment)

    def refund_payment(self, payment):
        with transaction.atomic():
            payment = CardPayment.objects.select_for_update().get(id=payment.id)
            payment_status = payment.get_status()
            if payment_status is CardPayment.Status.CANCELED:
                LOGGER.warning('payment already refunded: %s', payment.id)
                return
            elif payment_status is CardPayment.Status.AUTHORIZED:
                self._refund_authorized_payment(payment)
            elif payment_status is CardPayment.Status.CLEARED:
                self._refund_cleared_payment(payment)
            else:
                raise RuntimeError('unreachable: {}'.format(payment_status))

            self.sync_payment(payment)

    def _refund_authorized_payment(self, payment):
        assert payment.get_status() is CardPayment.Status.AUTHORIZED
        self._client.unhold_payment(
            uid=payment.user.uid,
            purchase_token=payment.purchase_token,
        )
        self.sync_payment(payment)

    def _refund_cleared_payment(self, payment):
        if payment.get_status() is CardPayment.Status.REFUNDED:
            LOGGER.warning('payment already refunded: %s', payment.id)
            return

        assert payment.get_status() is CardPayment.Status.CLEARED

        self._client.refund_payment(
            uid=payment.user.uid,
            purchase_token=payment.purchase_token,
        )
        self.sync_payment(payment)

    def sync_payment(self, payment):
        response = self._client.get_payment(
            uid=payment.user.uid,
            purchase_token=payment.purchase_token,
        )
        self.sync_payment_from_response(payment=payment, response=response)
        return response

    def sync_payment_from_response(self, payment, response):
        payment.status = CardPayment.Status(response['payment_status']).value

        if 'current_amount' in response:
            payment.amount = decimal.Decimal(response['current_amount'])
        else:
            payment.amount = decimal.Decimal(response['amount'])

        if 'orig_amount' in response:
            payment.orig_amount = decimal.Decimal(response['orig_amount'])
        else:
            payment.orig_amount = payment.amount

        if 'start_ts' in response:
            payment.started_at = self._parse_ts(response['start_ts'])
        if 'update_ts' in response:
            payment.updated_at = self._parse_ts(response['update_ts'])
        if 'final_status_ts' in response:
            payment.completed_at = self._parse_ts(response['final_status_ts'])

        payment.resp_code = response.get('payment_resp_code')
        if payment.resp_code:
            payment.resp_code = payment.resp_code[:32]

        payment.resp_desc = response.get('payment_resp_desc')
        if payment.resp_desc:
            payment.resp_desc = payment.resp_desc[:128]

        payment.save()

    def _parse_ts(self, ts):
        ts = float(ts)
        return datetime.datetime.fromtimestamp(ts, tz=pytz.UTC)

    def wait_for_completion(self, payments, timeout):
        for p in payments:
            assert isinstance(p, CardPayment)

        futures = []
        processed_payments = []
        with concurrent.futures.ThreadPoolExecutor(max_workers=len(payments)) as tp:
            for payment in payments:
                future = tp.submit(
                    self._do_wait_for_completion,
                    payment=payment,
                    timeout=timeout,
                )
                futures.append(future)

            for future, orig_payment in zip(futures, payments):
                try:
                    payment = future.result(timeout=timeout)
                except concurrent.futures.TimeoutError:
                    LOGGER.warning(
                        'payment failed to complete after %ss: %s',
                        timeout,
                        orig_payment.id,
                    )
                    payment = None
                processed_payments.append(payment)

        return processed_payments

    def _do_wait_for_completion(self, payment, timeout):
        start_time = time.time()
        ready_statuses = {CardPayment.Status.AUTHORIZED} | CardPayment.FINAL_STATUSES

        # Initialize payment outside of a transaction to avoid rolling it back after start.
        if payment.get_status() is CardPayment.Status.DRAFT:
            self.initialize_payment(payment)

        with transaction.atomic():
            payment = CardPayment.objects.select_for_update().get(id=payment.id)
            while time.time() - start_time < timeout:
                if payment.get_status() in ready_statuses:
                    break
                payment = self.process(payment)

        if payment.get_status() not in ready_statuses:
            raise RuntimeError

        return payment
