import collections
import datetime
import decimal
import logging
import operator

from django.utils import timezone

from cars.core.daemons import CarsharingDaemon
from ..models.bonus_account import BonusAccount
from ..models.bonus_payment import BonusPayment
from ..models.card_payment import CardPayment


LOGGER = logging.getLogger(__name__)


Interval = collections.namedtuple('Interval', ['label', 'timedelta'])


class BillingMonitoringDaemon(CarsharingDaemon):

    tick_interval = '*/3 * * * *'

    def get_distributed_lock_relative_path(self):
        return 'billing/locks/monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'billing.monitoring'

    def get_solomon_service(self):
        return 'billing'

    def _do_tick(self):
        exc = None

        try:
            self.monitor_bonus_payments()
        except Exception as e:
            exc = e
            LOGGER.exception('failed to monitor bonus payments')

        try:
            self.monitor_card_payments()
        except Exception as e:
            exc = e
            LOGGER.exception('failed to monitor card payments')

        if exc is not None:
            raise exc  # pylint: disable=raising-bad-type

    def monitor_bonus_payments(self):
        self.monitor_bonus_payments_bonuses_in_circulation()
        self.monitor_bonus_payments_amount()
        self.monitor_bonus_payments_count()

    def monitor_bonus_payments_bonuses_in_circulation(self):
        bonuses_in_circulation = decimal.Decimal(0)
        for account in BonusAccount.objects.all():
            bonuses_in_circulation += account.balance
        self.solomon.set_value(
            'bonus_payments.bonuses_in_circulation',
            float(bonuses_in_circulation),
        )

    def monitor_bonus_payments_amount(self):
        return self.do_monitor_bonus_payments(
            sensor='bonus_payments.amount',
            extractor=lambda payment: float(payment.amount),
        )

    def monitor_bonus_payments_count(self):
        return self.do_monitor_bonus_payments(
            sensor='bonus_payments.count',
            extractor=lambda payment: 1,
        )

    def do_monitor_bonus_payments(self, sensor, extractor):
        intervals = self._get_default_intervals()
        since = self._get_intervals_start_dt(intervals)

        payments = (
            BonusPayment.objects
            .filter(created_at__gte=since)
        )

        payments_ts = []
        for payment in payments:
            payments_ts.append((payment.created_at, extractor(payment)))

        self._monitor_timeseries(
            ts=payments_ts,
            intervals=intervals,
            sensor=sensor,
        )

    def monitor_card_payments(self):
        self.monitor_card_payments_count_by_status()
        self.monitor_card_payments_amount_by_status()

    def monitor_card_payments_count_by_status(self):
        return self.monitor_card_payments_by_status(
            sensor='card_payments.count_per_status',
            extractor=lambda payment: 1,
        )

    def monitor_card_payments_amount_by_status(self):
        return self.monitor_card_payments_by_status(
            sensor='card_payments.amount_per_status',
            extractor=lambda payment: float(payment.amount),
        )

    def monitor_card_payments_by_status(self, sensor, extractor):
        intervals = self._get_default_intervals()
        since = self._get_intervals_start_dt(intervals)

        payments_per_status = collections.defaultdict(list)
        for payment in CardPayment.objects.filter(created_at__gt=since):
            payments_per_status[payment.status].append(payment)

        for status, payments in payments_per_status.items():
            payments_ts = [(p.created_at, extractor(p)) for p in payments]
            default_labels = {
                'card_payment_status': status,
            }
            self._monitor_timeseries(
                ts=payments_ts,
                intervals=intervals,
                sensor=sensor,
                default_labels=default_labels,
            )

    def _monitor_timeseries(self, ts, intervals, sensor, default_labels=None):
        if default_labels is None:
            default_labels = {}

        values_per_interval = collections.defaultdict(lambda: 0)

        now = timezone.now()
        for dt, value in ts:
            for interval in intervals:
                if now - dt > interval.timedelta:
                    continue
                values_per_interval[interval] += value

        for interval in intervals:
            interval_value = values_per_interval[interval]
            labels = default_labels.copy()
            labels['interval'] = interval.label
            print(sensor, interval_value, labels)
            self.solomon.set_value(sensor, interval_value, labels=labels)

    def _get_default_intervals(self):
        intervals = [
            Interval(label=label, timedelta=datetime.timedelta(hours=hours))
            for label, hours in [
                ('1h', 1),
#                ('6h', 6),
#                ('1d', 24 * 1),
#                ('1w', 24 * 7),
                ('1m', 24 * 31),
            ]
        ]
        return intervals

    def _get_intervals_start_dt(self, intervals):
        greatest_interval = max(intervals, key=operator.attrgetter('timedelta'))
        start_dt = timezone.now() - greatest_interval.timedelta
        return start_dt
