from django.db import models

from cars.users.models.user import User


class BonusAccount(models.Model):

    user = models.OneToOneField(
        User,
        related_name='bonus_account',
        on_delete=models.CASCADE,
        db_index=False,
    )

    balance = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )

    generic_earned = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )
    generic_spent = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )

    registration_taxi_cashback_earned = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )
    registration_taxi_cashback_spent = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )

    @property
    def registration_taxi_cashback_balance(self):
        return self.registration_taxi_cashback_earned - self.registration_taxi_cashback_spent

    @property
    def generic_balance(self):
        return self.generic_earned - self.generic_spent

    class Meta:
        db_table = 'billing_bonus_account'
        indexes = [
            models.Index(
                fields=['user'],
                name='billing_bonus_account_user_idx',
            ),
        ]
        db_constraints = {
            'balance_check': (
                '''
                CHECK (
                  (
                    balance =
                      registration_taxi_cashback_earned - registration_taxi_cashback_spent
                      + generic_earned - generic_spent
                  )
                  AND
                  (registration_taxi_cashback_earned >= registration_taxi_cashback_spent)
                  AND
                  (generic_earned >= generic_spent)
                )
                '''
            ),
        }


class BonusAccountOperation(models.Model):

    bonus_account = models.ForeignKey(
        BonusAccount,
        related_name='operations',
        on_delete=models.CASCADE,
        db_index=False,
    )

    created_at = models.DateTimeField()
    created_by = models.ForeignKey(
        User,
        related_name='created_bonus_account_operations',
        on_delete=models.CASCADE,
        db_index=False,
    )

    amount = models.DecimalField(
        max_digits=10,
        decimal_places=2,
    )
    balance = models.DecimalField(
        max_digits=10,
        decimal_places=2,
    )
    nonce = models.CharField(max_length=64)

    comment = models.TextField()

    class Meta:
        db_table = 'billing_bonus_account_operation'
        indexes = [
            models.Index(
                fields=['bonus_account'],
                name='billing_bonus_account_operation_bonus_account_idx',
            ),
            models.Index(
                fields=['created_at'],
                name='billing_bonus_account_operation_created_at_idx',
            ),
            models.Index(
                fields=['created_by'],
                name='billing_bonus_account_operation_created_by_idx',
            ),
        ]
