import enum

from django.db import models

from cars.users.models.user import User
from ..iface.payment import IPayment


class CardPayment(IPayment, models.Model):

    class Status(enum.Enum):
        # Before Trust payment is created.
        DRAFT = 'draft'
        INVALID_PAYMENT_METHOD = 'invalid_payment_method'

        # After Trust payment is created.
        NOT_STARTED = 'not_started'
        STARTED = 'started'
        AUTHORIZED = 'authorized'
        NOT_AUTHORIZED = 'not_authorized'
        IN_PROGRESS = 'in_progress'
        CLEARED = 'cleared'
        CANCELED = 'canceled'
        REFUNDED = 'refunded'

    FINAL_STATUSES = {
        Status.INVALID_PAYMENT_METHOD,
        Status.NOT_AUTHORIZED,
        Status.CLEARED,
        Status.CANCELED,
        Status.REFUNDED,
    }

    user = models.ForeignKey(
        User,
        on_delete=models.CASCADE,
    )
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    orig_amount = models.DecimalField(max_digits=10, decimal_places=2, null=True)
    status = models.CharField(
        max_length=32,
        choices=[(x.value, x.name) for x in Status],
        default=Status.DRAFT.value,
    )
    paymethod_id = models.CharField(max_length=32)
    purchase_token = models.CharField(max_length=32, null=True)
    resp_code = models.CharField(max_length=32, null=True)
    resp_desc = models.CharField(max_length=128, null=True)

    created_at = models.DateTimeField()
    updated_at = models.DateTimeField()
    started_at = models.DateTimeField(null=True)
    completed_at = models.DateTimeField(null=True)

    class Meta:
        db_table = 'billing_card_payment'
        indexes = [
            models.Index(
                fields=['user'],
                name='billing_card_payment_user_idx',
            ),
            models.Index(
                fields=['status'],
                name='billing_card_payment_status_idx',
            ),
            models.Index(
                fields=['created_at'],
                name='billing_card_payment_created_at_idx',
            ),
            models.Index(
                fields=['started_at'],
                name='billing_card_payment_started_at_idx',
            ),
        ]

    def __str__(self):
        return '<CardPayment: status={}, amount={}, purchase_token={}, created_at={}>'.format(
            self.status,
            self.amount,
            self.purchase_token,
            self.created_at.isoformat(),
        )

    def get_status(self):
        return self.Status(self.status)

    def get_amount(self):
        return self.amount

    def get_generic_status(self):
        generic_status_map = {
            self.Status.DRAFT: self.GenericStatus.IN_PROGRESS,
            self.Status.NOT_STARTED: self.GenericStatus.IN_PROGRESS,
            self.Status.STARTED: self.GenericStatus.IN_PROGRESS,
            self.Status.IN_PROGRESS: self.GenericStatus.IN_PROGRESS,
            self.Status.AUTHORIZED: self.GenericStatus.SUCCESS,
            self.Status.CLEARED: self.GenericStatus.SUCCESS,
            self.Status.CANCELED: self.GenericStatus.REFUNDED,
            self.Status.REFUNDED: self.GenericStatus.REFUNDED,
            self.Status.INVALID_PAYMENT_METHOD: self.GenericStatus.ERROR,
            self.Status.NOT_AUTHORIZED: self.GenericStatus.ERROR,
        }

        generic_status = generic_status_map.get(self.get_status())
        if generic_status is None:
            raise RuntimeError('unreachable: {}'.format(self.get_status()))

        return generic_status
