# pylint: disable=redefined-outer-name
import collections
import datetime
import decimal

import furl
import pytz
from django.utils import timezone
from kubiki.util import make_requests_session

import cars.settings


class CalculatorClientBase:

    CarsharingBaseTariff = collections.namedtuple(
        'CarsharingBaseTariff',
        [
            'start_time',
            'ride_cost_per_minute',
            'parking_cost_per_minute',
            'free_parking',
        ],
    )

    CarsharingBaseTariffFreeParking = collections.namedtuple(
        'CarsharingBaseTariffFreeParking',
        [
            'end_date',
            'next_tariff',
        ],
    )


class CalculatorClient(CalculatorClientBase):

    def __init__(self, root_url, retries, pool_maxsize, timeout, verify_ssl):
        self._urls = CalculatorClientUrls(root_url=root_url)
        self._timeout = timeout
        self._verify_ssl = verify_ssl
        self._session = make_requests_session(retries=retries, pool_maxsize=pool_maxsize)

    @classmethod
    def from_settings(cls):
        settings = cars.settings.CALCULATOR['client']
        return cls(
            root_url=settings['root_url'],
            retries=settings['retries'],
            pool_maxsize=settings['pool_maxsize'],
            timeout=settings['timeout'],
            verify_ssl=settings['verify_ssl'],
        )

    def get_carsharing_base_tariffs(self, user, cars, dt=None, tz=None):
        """
        Pick carsharing base tariff.

        Args:
            user: User instance.
            cars: A list of Car instances.
            dt: Moment of time at which to pick the tariff.
            tz: timezone of datetime.time members of the response.

        Returns:
            CarsharingBaseTariff instance.
        """

        if dt is None:
            dt = timezone.now()

        cars_per_model = collections.defaultdict(list)
        for car in cars:
            cars_per_model[car.model_id].append(car)

        requests_data = []
        for model_code in cars_per_model:
            request_data = {
                'date': dt.timestamp(),
                'user_tags': user.tags or [],
                'is_plus_user': user.get_plus_status(),
                'car_model_code': model_code,
            }
            if tz is not None:
                request_data['timezone'] = tz.zone
            requests_data.append(request_data)

        data = {
            'requests': requests_data,
        }

        response = self._request(url=self._urls.carsharing_base_url, data=data)

        result = {}

        for request_data, result_data in zip(requests_data, response['results']):
            free_parking_data = result_data['free_parking']
            if free_parking_data is not None:
                end_date = pytz.utc.localize(
                    datetime.datetime.utcfromtimestamp(
                        free_parking_data['end_date']
                    )
                )
                next_tariff = self._parse_carsharing_base_tariff(
                    data=free_parking_data['next_tariff'],
                    free_parking=None,
                )
                free_parking = self.CarsharingBaseTariffFreeParking(
                    end_date=end_date,
                    next_tariff=next_tariff,
                )
            else:
                free_parking = None

            tariff = self._parse_carsharing_base_tariff(
                data=result_data,
                free_parking=free_parking,
            )

            for car in cars_per_model[request_data['car_model_code']]:
                result[car] = tariff

        return result

    def _request(self, url, data):
        r = self._session.post(
            url,
            json=data,
            timeout=self._timeout,
            verify=self._verify_ssl,
        )
        r.raise_for_status()
        return r.json()

    def _parse_carsharing_base_tariff(self, data, free_parking):
        start_time = self._parse_time(data['start_time'])
        ride_cost_per_minute = decimal.Decimal(data['ride_cost_per_minute'])
        parking_cost_per_minute = decimal.Decimal(data['parking_cost_per_minute'])
        tariff = self.CarsharingBaseTariff(
            start_time=start_time,
            ride_cost_per_minute=ride_cost_per_minute,
            parking_cost_per_minute=parking_cost_per_minute,
            free_parking=free_parking,
        )
        return tariff

    def _parse_time(self, date_str):
        if date_str is None:
            return None
        return datetime.datetime.strptime(date_str, '%H:%M:%S').time()


class CalculatorClientUrls:

    def __init__(self, root_url):
        self._root_furl = furl.furl(root_url)
        self.carsharing_base_url = self._root_furl.copy().add(path='carsharing/base/').url


class StubCalculatorClient(CalculatorClientBase):

    _default_tariff = None
    _tariff_per_car = {}

    @classmethod
    def from_settings(cls):
        return cls()

    @classmethod
    def set_carsharing_base_default_tariff(cls,
                                           ride_cost_per_minute,
                                           parking_cost_per_minute,
                                           start_time=None):
        if start_time is None:
            start_time = timezone.now().time()
        cls._default_tariff = cls.CarsharingBaseTariff(
            start_time=start_time,
            ride_cost_per_minute=ride_cost_per_minute,
            parking_cost_per_minute=parking_cost_per_minute,
            free_parking=None,
        )

    @classmethod
    def set_carsharing_base_tariff_per_car(cls,
                                           car,
                                           ride_cost_per_minute,
                                           parking_cost_per_minute,
                                           start_time=None):
        if start_time is None:
            start_time = timezone.now().time()
        cls._tariff_per_car[car] = cls.CarsharingBaseTariff(
            start_time=start_time,
            ride_cost_per_minute=ride_cost_per_minute,
            parking_cost_per_minute=parking_cost_per_minute,
            free_parking=None,
        )

    @classmethod
    def clear(cls):
        cls._default_tariff = None
        cls._tariff_per_car = {}

    def get_carsharing_base_tariffs(self, user, cars, dt=None, tz=None):  # pylint: disable=unused-argument
        if dt is None:
            dt = timezone.now()

        tariffs = {}
        for car in cars:
            tariff = self._tariff_per_car.get(car)
            if tariff is None:
                tariff = self._default_tariff
            tariffs[car] = tariff

        return tariffs
