import pytz
from rest_framework.serializers import (
    CharField, DecimalField, ListField,
    NullBooleanField, SerializerMethodField, TimeField, ValidationError,
)

from cars.carsharing.core.tariff_manager import CarsharingTariffPickerRequest
from cars.django.serializers import BaseSerializer, TimestampField


class CarsharingBaseRequestArgumentsSerializer(BaseSerializer):
    date = TimestampField()
    timezone = CharField(required=False)
    user_tags = ListField(child=CharField())
    car_model_code = CharField()
    is_plus_user = NullBooleanField(default=False, required=False)

    def validate_timezone(self, value):
        try:
            tz = pytz.timezone(value)
        except pytz.UnknownTimeZoneError:
            raise ValidationError
        return tz

    def to_internal_value(self, data):
        data = super().to_internal_value(data)
        return CarsharingTariffPickerRequest(
            date=data['date'],
            user_tags=data['user_tags'],
            car_model_code=data['car_model_code'],
            timezone=data.get('timezone'),
            is_plus_user=data.get('is_plus_user') or False,
        )


class CarsharingBaseArgumentsSerializer(BaseSerializer):
    requests = ListField(child=CarsharingBaseRequestArgumentsSerializer())


class CarsharingBaseTariffSerializer(BaseSerializer):
    start_time = TimeField()
    ride_cost_per_minute = DecimalField(max_digits=10, decimal_places=2)
    parking_cost_per_minute = DecimalField(max_digits=10, decimal_places=2)
    free_parking = SerializerMethodField()

    def get_free_parking(self, obj):
        if obj.free_parking is None:
            return None
        return CarsharingBaseTariffFreeParkingSerializer(obj.free_parking).data


class CarsharingBaseTariffFreeParkingSerializer(BaseSerializer):
    end_date = TimestampField()
    next_tariff = CarsharingBaseTariffSerializer()
