import logging

from django.shortcuts import get_object_or_404
from django.utils import timezone

import cars.settings
from ..models import CallcenterUserLink


LOGGER = logging.getLogger(__name__)


class ACLVerifier:

    def __init__(self, acl):
        self._acl = acl

    @classmethod
    def from_settings(cls):
        settings = cars.settings.CALLCENTER
        return cls(
            acl=settings['acl'],
        )

    def is_allowed(self, uid):
        return (self._acl is None or (self._acl and uid in self._acl))


class CallLinkAccessVerifier:

    def is_allowed(self, uid, call_id):
        if not call_id:
            return False

        LOGGER.info(
            'User with uid=%s visits the link with call_id=%s',
            str(uid),
            call_id
        )

        return self._is_assigned_to_link(call_id, uid)

    def _is_assigned_to_link(self, call_id, requester_uid):
        link = get_object_or_404(CallcenterUserLink, id=call_id)

        if link.available_until < timezone.now():
            return False, 'link.expired'
        if link.requester.uid != requester_uid:
            return False, 'link.uid_mismatch'

        return True, None


class CallcenterAuthentication:

    def __init__(self, acl_verifier, call_id_verifier):
        self._acl_verifier = acl_verifier
        self._call_id_verifier = call_id_verifier

    @classmethod
    def from_settings(cls):
        return cls(
            acl_verifier=ACLVerifier.from_settings(),
            call_id_verifier=CallLinkAccessVerifier(),
        )

    def is_allowed(self, uid, call_id=None):
        if self._acl_verifier is not None and not self._acl_verifier.is_allowed(uid):
            return False, 'user.not_in_acl'

        if self._call_id_verifier is not None:
            is_allowed, error_reason = self._call_id_verifier.is_allowed(uid, call_id)
            if not is_allowed:
                return False, error_reason

        return True, None
