import collections
import logging

from django.db import transaction
from django.shortcuts import get_object_or_404

import cars.settings

from cars.core.history import HistoryManager
from cars.core.util import datetime_helper, import_class
from cars.settings import CALLCENTER as settings

from cars.users.models import User

from ..models import CallCenterBlacklistedPhone, CallCenterBlacklistedPhoneHistory

LOGGER = logging.getLogger(__name__)


class PhoneBlacklistManager(object):
    BlackListEntry = collections.namedtuple('BlackListEntry', ('phone', 'blocked_until', 'comment'))

    BlackListHistoryEntry = collections.namedtuple(
        'BlackListHistoryEntry',
        ('action', 'performed_at', 'performed_by', 'phone', 'blocked_until', 'comment')
    )

    def __init__(self, tel_api_client):
        self._tel_api_client = tel_api_client

        self._history_manager = HistoryManager(
            CallCenterBlacklistedPhoneHistory,
            history_field_names={'event': 'history_id'}
        )

    @classmethod
    def from_settings(cls):
        tel_api_client_class = import_class(settings['api']['client_class'])
        tel_api_client = tel_api_client_class.from_settings()
        return cls(
            tel_api_client=tel_api_client,
        )

    def add_phone_to_blacklist(self, performed_by, phone, blocked_until=None, comment=''):
        is_blocked, existing_blocked_until = self.is_phone_blacklisted(phone=phone)

        if is_blocked:
            raise Exception('phone {} is already blacklisted'.format(phone))
        else:
            if existing_blocked_until is not None:
                LOGGER.info('removing expired block from blacklist: phone - {}'.format(phone))
                self.remove_phone_from_blacklist(performed_by, phone)

        self._tel_api_client.add_to_blacklist(phones=(phone, ))

        with transaction.atomic(savepoint=False):
            entry = CallCenterBlacklistedPhone.objects.create(
                phone=phone, blocked_until=blocked_until, comment=comment,
            )
            self._history_manager.add_entry(entry, str(performed_by.id), datetime_helper.utc_now())

    def remove_phone_from_blacklist(self, performed_by, phone):
        is_blocked, blocked_until = self.is_phone_blacklisted(phone=phone)

        if not is_blocked and blocked_until is None:
            raise Exception('cannot remove phone {} from blacklist as it does not present there'.format(phone))

        if is_blocked:
            # do not remove expired items from telephony
            self._tel_api_client.remove_from_blacklist(phones=(phone, ))

        with transaction.atomic(savepoint=False):
            entries = CallCenterBlacklistedPhone.objects.filter(phone=phone)

            performed_at = datetime_helper.utc_now()

            for entry in entries:
                self._history_manager.remove_entry(entry, str(performed_by.id), performed_at)
                entry.delete()

    def is_phone_blacklisted(self, *, user_id=None, phone=None):
        if not ((user_id is None) ^ (phone is None)):
            raise Exception('only one filter must be applied to check blacklisted user')

        if user_id is not None:
            phone = get_object_or_404(User, id=user_id).phone

        blacklisted_entry = (
            CallCenterBlacklistedPhone.objects.filter(phone=phone).first()
        )

        if blacklisted_entry is not None:
            blocked_until = blacklisted_entry.blocked_until
            is_blocked = (blocked_until is None or datetime_helper.utc_now() <= blocked_until)
        else:
            is_blocked, blocked_until = False, None

        return is_blocked, blocked_until

    def get_phone_blacklist_history(self, limit=None, **kwargs):
        filters = self._make_history_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        history_entries = (
            CallCenterBlacklistedPhoneHistory.objects
            .select_related('history_user')
            .filter(**filters)
            .order_by('-history_timestamp')
            [:limit]
        )

        return history_entries

    def _make_history_filters(self, phone=None, since=None, until=None):
        filters = {}

        if since is not None:
            filters['history_timestamp__gte'] = since

        if until is not None:
            filters['history_timestamp__lt'] = until

        if phone is not None:
            filters['phone'] = phone

        return filters
