import collections
import concurrent.futures
import logging

from django.db import transaction
from django.db.models import Q

import cars.settings

from cars.carsharing.models.car import Car
from cars.core.saas_drive import SaasDrive
from cars.core.util import import_class, phone_number_helper, datetime_helper
from cars.settings import CALLCENTER as settings
from cars.users.models.user import User
from cars.users.serializers import UserSerializer

from ..models.call_priority import CallPriorityUser
from .blacklist_manager import PhoneBlacklistManager


LOGGER = logging.getLogger(__name__)


class CallPriorityManager:
    PrioritizedUserInfo = collections.namedtuple(
        'PrioritizedUserInfo', ('call_priority_user', 'user', 'user_id', 'user_phone')
    )

    # time after last priority session to disable call priority
    PRIORITY_EXPIRE_SECONDS = 3600 * 24
    # non-priority sessions after last priority order to disable call priority
    PRIORITY_EXPIRE_ORDERS = 3
    # number of car clients to analyze
    #   use specific count for simplicity processing paginated results
    #   and ensure that all entries have been assigned in case of shutdown
    PRIORITY_SESSIONS_TO_CHECK = 20

    WORK_THREAD_COUNT = 10
    WORK_THREAD_TIMEOUT_S = None

    def __init__(self, tel_api_client, saas_client, oauth_token):
        self._tel_api_client = tel_api_client
        self._saas_client = saas_client
        self._blacklist_manager = PhoneBlacklistManager.from_settings()
        self._oauth_token = oauth_token

    @classmethod
    def from_settings(cls):
        tel_api_client_class = import_class(settings['api']['client_class'])
        tel_api_client = tel_api_client_class.from_settings()
        return cls(
            tel_api_client=tel_api_client,
            saas_client=SaasDrive.from_settings(
                version='v5.0.0',
                prestable=False,
            ),
            oauth_token=settings['saas_history_token']
        )

    def _get_premium_car_ids(self):
        return (
            Car.objects.using(cars.settings.DB_RO_ID)
            .filter(
                Q(model__code__startswith='porsche') |
                Q(model__code__startswith='range_rover') |
                Q(model__code__startswith='nissan_leaf') |
                Q(model__code__startswith='jeep') |
                Q(model__code__startswith='mustang')
            )
            .values_list('id', flat=True)
        )

    def _do_session_has_priority(self, session_dict):
        return session_dict['car']['model_id'].lower().startswith(('porsche', 'range_rover', 'mustang', 'jeep', 'nissan_leaf'))

    def _is_segment_finished(self, segment, default=True):
        return segment.get('meta', {}).get('finished', default)

    def _get_segment_finish_time(self, segment):
        finish_time = segment.get('meta', {}).get('finish', None)

        if finish_time is None:
            finish_time = segment.get('finish', None)

        return finish_time

    def _do_user_has_priority(self, user_id):
        if not user_id:
            return False

        min_start_time = int(datetime_helper.timestamp_now() - self.PRIORITY_EXPIRE_SECONDS)

        sessions = self._saas_client.get_user_history(
            user_id=user_id,
            numdoc=self.PRIORITY_EXPIRE_ORDERS,
            oauth_token=self._oauth_token,
            since=min_start_time
        )['sessions']

        if not sessions:
            return False

        if (
                self._is_segment_finished(sessions[0]['segment']) and
                self._get_segment_finish_time(sessions[0]['segment']) is not None
        ):
            has_priority = any(
                self._do_session_has_priority(s) for s in sessions
                if self._get_segment_finish_time(s['segment']) is not None and self._get_segment_finish_time(s['segment']) > min_start_time
            )
        else:
            # check current ride
            has_priority = self._do_session_has_priority(sessions[0])

        return has_priority

    def _iter_last_priority_cars_users(self):
        premium_car_ids = self._get_premium_car_ids()

        with concurrent.futures.ThreadPoolExecutor(max_workers=self.WORK_THREAD_COUNT) as executor:
            for sessions in executor.map(
                    self._get_car_sessions_to_check, premium_car_ids, timeout=self.WORK_THREAD_TIMEOUT_S
            ):
                for s in sessions:
                    user_meta_info = s['user_details']
                    user_id = user_meta_info['id']
                    user_phone = user_meta_info['setup']['phone']['number']
                    yield user_id, user_phone

    def _get_car_sessions_to_check(self, car_id):
        sessions = self._saas_client.get_car_history(
            car_id=car_id,
            numdoc=self.PRIORITY_SESSIONS_TO_CHECK,
            oauth_token=self._oauth_token,
        )['sessions']
        return sessions

    def add_call_priority(self):
        """Create CallPriorityUser entries and add priority in telephony"""
        registered_priority_user_ids = {item.user_id for item in self.iter_prioritized_user_info()}

        priority_user_ids_to_check = {
            user_id for user_id, _ in self._iter_last_priority_cars_users()
            if user_id not in registered_priority_user_ids
        }

        LOGGER.info('there are {} priority users to check'.format(len(priority_user_ids_to_check)))

        call_priority_users_to_create = []
        priority_phones_to_register = set()
        blocked_phones = set()

        for user_id in self._filter_prioritized_users(priority_user_ids_to_check):
            # ensure that user phone is correct
            raw_user_phone = User.objects.using(cars.settings.DB_RO_ID).filter(id=user_id).get().phone
            user_phone = phone_number_helper.normalize_phone_number(
                str(raw_user_phone) if raw_user_phone is not None else None
            )

            if user_phone is not None:
                is_blocked, _ = self._blacklist_manager.is_phone_blacklisted(phone=user_phone)

                if not is_blocked:
                    priority_phones_to_register.add(user_phone)

                    call_priority_user = CallPriorityUser(user_id=user_id)
                    call_priority_users_to_create.append(call_priority_user)
                else:
                    blocked_phones.add(user_phone)
            else:
                LOGGER.error('cannot assign priority to user {} due to bad phone number'.format(user_id))

        if blocked_phones:
            LOGGER.info('phones {} are blocked and priority will not be added'.format(list(blocked_phones)))

        LOGGER.info('there are {} priority users to add'.format(len(call_priority_users_to_create)))

        with transaction.atomic(savepoint=False):
            CallPriorityUser.objects.bulk_create(call_priority_users_to_create)
            self._tel_api_client.add_call_priority(phones=priority_phones_to_register)

    def _filter_prioritized_users(self, user_ids):
        with concurrent.futures.ThreadPoolExecutor(max_workers=self.WORK_THREAD_COUNT) as executor:
            for user_id, has_priority in zip(
                    user_ids,
                    executor.map(self._do_user_has_priority, user_ids, timeout=self.WORK_THREAD_TIMEOUT_S)
            ):
                if has_priority:
                    yield user_id

    def delete_call_priority(self):
        """Delete expired registered CallPriorityUser entries"""
        call_priority_users_to_delete = []
        priority_phones_to_delete = set()

        for info_item in self.iter_prioritized_user_info():
            if not self._do_user_has_priority(info_item.user_id):
                priority_phones_to_delete.add(info_item.user_phone)
                call_priority_users_to_delete.append(info_item.call_priority_user)

        with transaction.atomic(savepoint=False):
            self._tel_api_client.delete_call_priority(phones=priority_phones_to_delete)

            for cpu in call_priority_users_to_delete:
                cpu.delete()

    def sync_call_priority(self):
        """Sync telephony priority with registered ones by changing telephony priority queues"""
        registered_call_priority_phones = {item.user_phone for item in self.iter_prioritized_user_info()}
        telephony_call_priority_phones = set(self._tel_api_client.list_call_priority_phones())

        call_priority_phones_to_add = registered_call_priority_phones - telephony_call_priority_phones
        call_priority_phones_to_delete = telephony_call_priority_phones - registered_call_priority_phones

        if call_priority_phones_to_add or call_priority_phones_to_delete:
            LOGGER.error('diff in call priority phones; registered phones: {}; telephony phones: {}'.format(
                ','.join(registered_call_priority_phones),
                ','.join(telephony_call_priority_phones),
            ))
            self._tel_api_client.delete_call_priority(phones=call_priority_phones_to_delete)
            self._tel_api_client.add_call_priority(phones=call_priority_phones_to_add)

    def is_user_prioritized(self, *, user_id=None, phone=None):
        if not ((user_id is None) ^ (phone is None)):
            raise Exception('only one filter must be applied to check prioritized user')

        if phone is not None:
            user = (
                User.objects.using(cars.settings.DB_RO_ID).filter(phone=phone)
                .order_by('status')  # make "active" first if exists
                .first()
            )
            user_id = user.id if user is not None else None

        is_prioritized = CallPriorityUser.objects.filter(user_id=user_id).exists()
        return is_prioritized

    def iter_prioritized_user_info(self):
        for cpu in CallPriorityUser.objects.select_related('user').only('user'):
            user = cpu.user
            user_id = str(user.id)

            if user.phone is not None:
                user_phone = phone_number_helper.normalize_phone_number(str(user.phone))
            else:
                user_phone = None

            yield self.PrioritizedUserInfo(cpu, user, user_id, user_phone)

    def get_prioritized_formatted_user_entries(self):
        return [UserSerializer(item.user).data for item in self.iter_prioritized_user_info()]
