import datetime
import enum
import functools
import logging
import time

import cars.settings

from django.db import transaction

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.solomon import SolomonHelper
from cars.core.util import datetime_helper
from cars.users.models.user import User

from ..core.staff_helper import StaffInfoHelper
from ..serializers.user import RequestUserSerializer, StaffUserEntrySerializer
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter


LOGGER = logging.getLogger(__name__)


class CallRegistrar(object):
    MAX_ACTIVE_CALL_SHOW_TIME = datetime.timedelta(minutes=10)
    POLL_DELAY_S = 1
    POLL_TIMEOUT_S = 15

    saas_client = SaasDriveAdminClient.from_settings()
    staff_info_helper = StaffInfoHelper.make_default()
    solomon_helper = SolomonHelper('request_aggregator', 'call_registration')

    class Action(enum.Enum):
        START = 'start'
        CONNECT = 'connect'
        FINISH = 'finish'

    def __init__(self, denied_operator_view_roles):
        self._denied_operator_view_roles = denied_operator_view_roles

    @classmethod
    def from_settings(cls):
        denied_operator_view_roles = cars.settings.REQUEST_AGGREGATOR['callcenter']['denied_operator_view_roles']
        return cls(denied_operator_view_roles=denied_operator_view_roles)

    def register_call(self, action, from_number, to_number):
        if action == self.Action.START:
            self._start_call(from_number, to_number)
        elif action == self.Action.CONNECT:
            self._connect_call(from_number, to_number)
        elif action == self.Action.FINISH:
            self._finish_call(from_number, to_number)
        else:
            raise ValueError('Unknown action: {}'.format(action))

    def _start_call(self, from_number, to_number):
        from_user = User.objects.filter(phone=from_number).first()

        if from_user is None:
            LOGGER.error('user with phone {} not found'.format(from_number))

        to_user_binding = self.staff_info_helper.get_agent_entry(work_phone=to_number)

        if to_user_binding is None:
            to_user = None
            LOGGER.error('operator with phone {} not found'.format(to_number))
        else:
            to_user = to_user_binding.user

        # note: multiple assignments may present in the table due to performance reason

        CallAssignment.objects.create(
            from_number=from_number,
            to_number=to_number,
            from_user=from_user,
            to_user=to_user,
        )

    def _connect_call(self, from_number, to_number):
        current_time = datetime_helper.utc_now()

        with transaction.atomic(savepoint=False):
            call_entries = CallAssignment.objects.select_for_update().filter(
                from_number=from_number,
                to_number=to_number
            )

            for call_entry in call_entries:
                call_entry.connected_at = current_time
                call_entry.save()

    def _finish_call(self, from_number, to_number):
        assignments = CallAssignment.objects.filter(
            from_number=from_number,
            to_number=to_number
        )

        with transaction.atomic(savepoint=False):
            for assignment in assignments:
                assignment.delete()

    def get_active_call(self, agent_entry):
        active_call = (
            CallAssignment.objects
            .prefetch_related('from_user')
            .filter(to_user=agent_entry)
            .order_by('-id')
            .first()
        )

        if (
                active_call is not None and
                active_call.added_at + self.MAX_ACTIVE_CALL_SHOW_TIME >= datetime_helper.utc_now()
        ):
            return active_call

        return None

    def get_call_assignment(self, agent_entry, actual_data=None):
        active_call = self.get_active_call(agent_entry)
        is_same_call = self._is_same_call(active_call, actual_data)

        if not is_same_call and active_call is not None:
            self._report_lag(active_call)

        data = self._serialize_active_call(active_call)
        return data

    def poll_call_assignment(self, agent_entry, actual_data=None):
        time_start = time.time()

        data = None

        while time_start + self.POLL_TIMEOUT_S > time.time():
            active_call = self.get_active_call(agent_entry)
            is_same_call = self._is_same_call(active_call, actual_data)

            if data is None or not is_same_call:
                data = self._serialize_active_call(active_call)

            if is_same_call:
                time.sleep(self.POLL_DELAY_S)
            else:
                if active_call is not None:
                    self._report_lag(active_call)
                break

        return data

    def _is_same_call(self, active_call, actual_data):
        has_active_call = (active_call is not None)
        has_actual_data = (actual_data is not None and 'phone' in actual_data)

        if has_active_call and has_actual_data:
            is_same_call = (str(active_call.from_number) == str(actual_data['phone']))  # valid for None values too
        else:
            is_same_call = not (has_actual_data ^ has_active_call)

        return is_same_call

    def _report_lag(self, active_call):
        connection_time = active_call.connected_at
        call_id = active_call.call_id

        if connection_time is None:
            LOGGER.error('connection time for call id {} is not set'.format(call_id))
            self.solomon_helper.increment_counter('notification_lag_no_connect_time')
        elif active_call.reported_at is not None:
            LOGGER.warning('notification lag has been already reported for call id {}'.format(call_id))
        else:
            notification_lag = (datetime_helper.utc_now() - connection_time).total_seconds()

            if notification_lag < 0:
                LOGGER.error('notification lag is negative for call id {}: {}'.format(call_id, notification_lag))
                self.solomon_helper.increment_counter('notification_lag_is_negative')
            else:
                LOGGER.info('notification lag for call id {} is {:.6f}s'.format(call_id, notification_lag))
                self.solomon_helper.report_value('notification_lag', notification_lag)

                # notice: call entry may be duplicated
                active_call.reported_at = datetime_helper.utc_now()
                active_call.save()

    def get_all_call_assignments(self, request=None):
        active_calls = (
            CallAssignment.objects
            .using(cars.settings.DB_RO_ID)
            .prefetch_related('from_user', 'to_user')
            .order_by('-added_at', '-connected_at')
        )

        assigned_calls = []

        _processed_call_ids = set()
        _processed_operators = set()

        has_access_to_operator = not self.saas_client.check_user_role(
            *self._denied_operator_view_roles, request=request, require_all=False
        )
        has_access_to_cc_name = not self.saas_client.check_user_role(
            'team_adm_level3', request=request, require_all=False
        )

        for call in active_calls:
            if (
                    (call.call_id is None or call.call_id not in _processed_call_ids) and
                    (call.to_user is not None and call.to_user.id not in _processed_operators)
            ):
                serialized_call = self._serialize_active_call(
                    call,
                    add_operator=True,
                    has_access_to_operator=has_access_to_operator,
                    has_access_to_cc_name=has_access_to_cc_name
                )
                assigned_calls.append(serialized_call)

                _processed_operators.add(call.to_user.id)
                _processed_call_ids.add(call.call_id)

        data = {'data': assigned_calls}
        return data

    @functools.lru_cache(maxsize=256)
    def _serialize_active_call(
            self, active_call, add_operator=False, has_access_to_operator=True, has_access_to_cc_name=True
    ):
        data = {}

        if active_call is None:
            return data

        call_center = active_call.call_center if has_access_to_cc_name else ''

        formatted_from_user = RequestUserSerializer(active_call.from_user).data

        if formatted_from_user['phone'] is None:
            formatted_from_user['phone'] = str(active_call.from_number)

        timestamp_start = active_call.added_at.timestamp()

        time_connect = active_call.connected_at
        timestamp_connect = time_connect.timestamp() if time_connect is not None else None

        data = {
            'call_center': call_center,
            'started': timestamp_start,
            'connected': timestamp_connect,
            'from_user': formatted_from_user,
        }

        request = {}

        if call_center and active_call.call_id:
            if call_center == CallAssignmentCallCenter.AudioTele.value:
                request['origin'] = 'AUDIOTELE_REALTIME_INCOMING'  # RequestOriginType.AUDIOTELE_REALTIME_INCOMING
                request['id'] = active_call.call_id
            elif call_center == CallAssignmentCallCenter.Yandex.value:
                request['origin'] = 'CC_INTERNAL_REALTIME_INCOMING'  # RequestOriginType.CC_INTERNAL_REALTIME_INCOMING
                request['id'] = active_call.call_id
            else:
                pass

        data['request'] = request

        if add_operator:
            entry_to_format = None
            to_user_id = active_call.to_user_id

            if to_user_id is not None and has_access_to_operator:
                to_user_staff_entry = self.staff_info_helper.get_agent_entry(user_id=str(to_user_id))
                entry_to_format = to_user_staff_entry if to_user_staff_entry is not None else active_call.to_user

            to_user_formatted_entry = StaffUserEntrySerializer(entry_to_format).data
            data['to_user'] = to_user_formatted_entry

        return data
