import collections
import datetime
import json
import logging
import operator

from django.db import transaction

import cars.settings
from cars.core.util import datetime_helper
from cars.core.settings import SettingsManager
from cars.core.history import HistoryManager
from cars.core.telephony import TelephonyCallCenter, TelephonySettingsHelper
from cars.callcenter.models import CallCenterSettings, CallCenterSettingsHistory

LOGGER = logging.getLogger(__name__)


class CallCenterSettingsManager(object):
    def __init__(self, default_performer_id):
        self._history_manager = HistoryManager(CallCenterSettingsHistory)
        self._default_performer_id = default_performer_id

    @classmethod
    def from_settings(cls):
        default_performer_id = cars.settings.CALLCENTER['default_performer_id']
        return cls(default_performer_id=default_performer_id)

    def exists(self, setting_key):
        return CallCenterSettings.objects.using(cars.settings.DB_RO_ID).filter(setting_key=setting_key).exists()

    def get(self, setting_key):
        setting_instance = CallCenterSettings.objects.filter(setting_key=setting_key).first()
        value = setting_instance.setting_value if setting_instance is not None else None
        return value

    def upsert(self, setting_key, setting_value, performer=None, *, only_if_differ=False):
        operator_id = self._get_operator_id(performer)

        with transaction.atomic(savepoint=False):
            entry = CallCenterSettings.objects.select_for_update().filter(setting_key=setting_key).first()

            if entry is None:
                entry = CallCenterSettings.objects.create(
                    setting_key=setting_key, setting_value=setting_value
                )
                self._history_manager.add_entry(entry, operator_id)
            else:
                if not only_if_differ or entry.setting_value != setting_value:
                    entry.setting_value = setting_value
                    entry.save()

                    self._history_manager.update_entry(entry, operator_id)

        return entry

    def remove(self, setting_key, performer=None):
        operator_id = self._get_operator_id(performer)

        with transaction.atomic(savepoint=False):
            entries_to_remove = CallCenterSettings.objects.select_for_update().filter(setting_key=setting_key)

            for entry in entries_to_remove:
                self._history_manager.remove_entry(entry, operator_id)
                entry.delete()

    def _get_operator_id(self, performer=None):
        operator_id = str(performer.id) if performer is not None else self._default_performer_id
        return operator_id


class LoadBalanceHelper(object):
    SETTING_KEY = 'cc_load_balance'
    SUPPORT_SETTING_KEY = 'support.cc.load_balance'

    REQUIRED_VALUES_TOTAL = 100

    CallCenter = TelephonyCallCenter

    ADMISSIBLE_CC_NAMES = set(x.name for x in CallCenter)

    def __init__(self):
        self._cc_settings_manager = CallCenterSettingsManager.from_settings()
        self._settings_manager = SettingsManager.from_settings()

        self._telephony_helper = TelephonySettingsHelper.from_settings()

    @classmethod
    def from_settings(cls):
        return cls()

    def _convert_to_public(self, internal_value_mapping):
        # convert telephony internal cc name to public name
        value_mapping = {}
        for key, value in internal_value_mapping.items():
            try:
                key = self.CallCenter(key).name
            except ValueError:
                pass  # actually unknown name, to be added

            value_mapping[key] = value
        return value_mapping

    def _convert_to_internal(self, value_mapping):
        # convert public name to telephony internal cc name
        internal_value_mapping = {self.CallCenter[key].value: value for key, value in value_mapping.items()}
        return internal_value_mapping

    def check_values(self, settings):
        admissible_cc_names = self.ADMISSIBLE_CC_NAMES

        if any(key not in admissible_cc_names for key in settings):
            raise Exception(
                'call center name must be one of {}, but given {}'.format(admissible_cc_names, list(settings))
            )

        required_total = self.REQUIRED_VALUES_TOTAL
        values_total = sum(settings.values())

        if values_total != required_total:
            raise Exception('call center loads must sum to {} (currently - {})'.format(required_total, values_total))

    def _convert_calendar(self, value_mapping, converter):
        updated_value_mapping = {}

        for day, day_settings in value_mapping.items():
            updated_value_mapping[day] = {}

            for interval, interval_settings in day_settings.items():
                updated = converter(interval_settings)
                updated_value_mapping[day][interval] = updated

        return updated_value_mapping

    def check_calendar_values(self, settings):
        if (
                not isinstance(settings, collections.Mapping) or
                set(settings.keys()) ^ set(str(x) for x in range(7))
        ):
            raise Exception('day keys are invalid')

        for day, day_settings in settings.items():
            for interval, interval_settings in day_settings.items():
                try:
                    datetime.datetime.strptime(interval, "%H:%M")
                except (TypeError, ValueError):
                    raise Exception('invalid time interval {} for day {}'.format(interval, day))

                try:
                    self.check_values(interval_settings)
                except Exception:
                    raise

    def get_current_value(self, settings):
        now = datetime_helper.now()  # UTC+3 time, consider to use arbitrary timezone
        day_settings = settings[str(now.weekday())]

        key_getter = operator.attrgetter('hour', 'minute')
        sorted_day_interval_keys = sorted(key_getter(datetime.datetime.strptime(k, "%H:%M")) for k in day_settings)

        target_interval = None
        now_key = key_getter(now)

        for interval_key in sorted_day_interval_keys:
            if now_key >= interval_key:
                target_interval = interval_key
            else:
                break

        target_interval_settings = day_settings['{:02d}:{:02d}'.format(*target_interval)]
        return target_interval_settings

    def _get_calendar_setting_name(self, application_source):
        return '{}.{}'.format(self.SUPPORT_SETTING_KEY, application_source)

    def _get_calendar_setting_value(self, application_source):
        setting_key = self._get_calendar_setting_name(application_source)
        target_setting_value = self._settings_manager.get_value(setting_key)
        internal_value_mapping = json.loads(target_setting_value)
        return internal_value_mapping

    def _set_calendar_setting_value(self, application_source, internal_value_mapping, performer):
        target_setting_value = json.dumps(internal_value_mapping, sort_keys=True)
        setting_key = self._get_calendar_setting_name(application_source)
        performer_id = str(performer.id)
        self._settings_manager.set_value(setting_key, target_setting_value, performer_id, only_if_differ=True)

    def get_calendar_values(self, application_source):
        internal_value_mapping = self._get_calendar_setting_value(application_source)
        settings = self._convert_calendar(internal_value_mapping, self._convert_to_public)
        return settings

    def set_calendar_values(self, application_source, settings, performer=None):
        assert isinstance(settings, collections.Mapping)
        internal_value_mapping = self._convert_calendar(settings, self._convert_to_internal)
        self._set_calendar_setting_value(application_source, internal_value_mapping, performer)
        self.update_settings_from_calendar(application_source, internal_value_mapping)
        return settings

    def update_settings_from_calendar(self, application_source, internal_value_mapping=None):
        if internal_value_mapping is None:
            internal_value_mapping = self._get_calendar_setting_value(application_source)

        current_interval_settings = self.get_current_value(internal_value_mapping)
        self._telephony_helper.set_settings(current_interval_settings, application_source)
        return current_interval_settings
