import collections
import logging

from django.db import transaction

from cars.core.util import datetime_helper

from ..models import CallCenterStaffEntry, CallCenterStaffStatAccessEntry
from .staff_helper import StaffInfoHelper


LOGGER = logging.getLogger(__name__)


class StaffStatAccessHelper(object):
    StaffStatAccessEntry = collections.namedtuple(
        'StaffStatAccessEntry', ('username', 'print_name', 'work_phone', 'department_url')
    )

    staff_helper = StaffInfoHelper.make_default()

    def _make_staff_stat_access_entry(self, staff_entry):
        return self.StaffStatAccessEntry(
            username=staff_entry.username,
            print_name=staff_entry.print_name,
            work_phone=staff_entry.work_phone,
            department_url=staff_entry.department_url,
        )

    def get_all_entries(self, *, department_url_prefix=None):
        raw_entries = (
            CallCenterStaffStatAccessEntry.objects
            .select_related('staff_entry')
            .all()
        )

        if department_url_prefix is not None:
            raw_entries = raw_entries.filter(staff_entry__department_url__startswith=department_url_prefix)

        entries = [self._make_staff_stat_access_entry(e.staff_entry)._asdict() for e in raw_entries]
        data = {'data': entries, 'count': len(entries)}
        return data

    def get_entries_without_access(self, *, department_url_prefix=None):
        raw_entries = (
            CallCenterStaffStatAccessEntry.objects
            .select_related('staff_entry')
            .filter(outgoing_stat_granted_at__isnull=True)
        )

        if department_url_prefix is not None:
            raw_entries = raw_entries.filter(staff_entry__department_url__startswith=department_url_prefix)

        entries = [self._make_staff_stat_access_entry(e.staff_entry)._asdict() for e in raw_entries]
        data = {'data': entries, 'count': len(entries)}
        return data

    def update_stat_access_entries(self):
        all_entries = CallCenterStaffEntry.objects.filter(work_phone__isnull=False)
        existing_entries = set(CallCenterStaffStatAccessEntry.objects.values_list('staff_entry', flat=True))

        entries_to_add = [
            CallCenterStaffStatAccessEntry(staff_entry=e)
            for e in all_entries if e.id not in existing_entries
        ]

        CallCenterStaffStatAccessEntry.objects.bulk_create(entries_to_add)

    def bulk_update_outgoing_stat_access(self, data):
        if isinstance(data.get('data'), list):
            entries = data['data']
        else:
            entries = [data]

        with transaction.atomic(savepoint=False):
            for e in entries:
                work_phone = int(e['work_phone'])

                granted_at = refused_at = None

                if 'timestamp_granted' in e:
                    granted_at = datetime_helper.timestamp_to_datetime(e['timestamp_granted'])

                if 'timestamp_refused' in e:
                    refused_at = datetime_helper.timestamp_to_datetime(e['timestamp_refused'])

                if not ((granted_at is not None) ^ (refused_at is not None)):
                    raise Exception('either granted date or refuse date is accepted')

                self.update_outgoing_stat_access_granted_date(work_phone, granted_at, refused_at)

    def update_outgoing_stat_access_granted_date(self, work_phone, granted_at=None, refused_at=None):
        staff_entry = self.staff_helper.get_agent_entry(work_phone=work_phone)

        if staff_entry is not None:
            entry = CallCenterStaffStatAccessEntry.objects.get(staff_entry=staff_entry)

            if granted_at is not None:
                entry.outgoing_stat_granted_at = granted_at
            else:  # refused_at is not None
                entry.outgoing_stat_refused_at = refused_at

            entry.save()

        else:
            LOGGER.error('no agent with phone {} found to update stat access date'.format(work_phone))
