import collections
import datetime
import logging
import re
import threading

from django.db import transaction

import cars.settings

from cars.core.util import datetime_helper, phone_number_helper
from cars.core.telephony import TelephonyHelperBase
from cars.settings import STAFF as settings
from cars.users.models import User

from ..serializers.user import StaffUserEntrySerializer
from ..models import CallCenterStaffEntry


LOGGER = logging.getLogger(__name__)


class StaffInfoCache(object):
    STOP_WAIT_TIMEOUT_S = 0.5

    SOFT_TTL = datetime.timedelta(hours=6)
    HARD_TTL = datetime.timedelta(hours=12)

    CacheEntry = collections.namedtuple('CacheEntry', ('entry', 'updated_at'))

    FIELDS_TO_CACHE = (
        'id', 'uid', 'email', 'username', 'external_username', 'work_phone',
        'user_id', 'yang_worker_id', 'first_name', 'last_name'
    )

    absent_object = object()

    def __init__(self):
        self._cache_mapping = {}  # (filter_key, filter_value): CacheEntry

        self._hard_invalidation_time = datetime_helper.utc_now()

        self._invalidation_thread = None
        self._invalidation_stop_event = threading.Event()

    @property
    def active(self):
        return self._invalidation_thread is not None and self._invalidation_thread.is_alive()

    @property
    def enabled(self):
        return settings['cache_invalidation_enabled']

    def add(self, entry, *, filter_key=None, filter_value=None):
        cache_entry = self.CacheEntry(entry=entry, updated_at=datetime_helper.utc_now())

        if filter_key is not None:
            field_filters = [(filter_key, filter_value)]
        else:
            field_filters = [(field_key, getattr(entry, field_key)) for field_key in self.FIELDS_TO_CACHE]

        for field_key, field_value in field_filters:
            field_value = self._coerce_filter_value(field_value)
            if field_value is not None:
                self._cache_mapping[(field_key, field_value)] = cache_entry

    def get(self, filter_key, filter_value):
        cached_value = self.absent_object

        if filter_value is not None:
            filter_value = self._coerce_filter_value(filter_value)

            cache_entry = self._cache_mapping.get((filter_key, filter_value), self.absent_object)

            if cache_entry is not self.absent_object:
                cached_value = cache_entry.entry

        return cached_value

    def _coerce_filter_value(self, filter_value):
        if filter_value is not None:
            filter_value = str(filter_value)
        return filter_value

    def invalidate_async(self):
        if not self.enabled:
            return

        if self.active:
            raise RuntimeError('invalidation is in progress')

        self._invalidation_stop_event.clear()

        self._invalidation_thread = threading.Thread(target=self._invalidate, daemon=True)
        self._invalidation_thread.start()

    def stop_invalidation(self):
        self._invalidation_stop_event.set()
        self._invalidation_thread.join(timeout=self.STOP_WAIT_TIMEOUT_S)

        if not self._invalidation_thread.is_alive():
            self._invalidation_thread = None

    def _invalidate(self):
        while True:
            self._invalidate_hard()  # soft invalidation is not used separately

            stop_requested = self._invalidation_stop_event.wait(self.HARD_TTL.total_seconds())
            if stop_requested:
                break

    def _invalidate_hard(self):
        LOGGER.debug('cache invalidation requested (hard)')

        try:
            for entry in self._iter_all_entries():
                self.add(entry)

        except Exception as exc:
            LOGGER.exception('error invalidating cache (hard)')

        else:
            self._hard_invalidation_time = datetime_helper.utc_now()
            LOGGER.debug('cache has been invalidated successfully (hard)')

        self._invalidate_soft()

    def _invalidate_soft(self):
        LOGGER.debug('cache invalidation requested (soft)')

        try:
            now = datetime_helper.utc_now()

            keys_to_pop = []

            for key, value in self._cache_mapping.items():
                if value.updated_at + self.SOFT_TTL >= now:
                    keys_to_pop.append(key)

            for key in keys_to_pop:
                self._cache_mapping.pop(key, None)

        except Exception as exc:
            LOGGER.exception('error invalidating cache (soft)')

        else:
            LOGGER.debug('cache has been invalidated successfully (soft)')

    def _iter_all_entries(self):
        entries = CallCenterStaffEntry.objects.all()
        return entries


class StaffInfoHelper(TelephonyHelperBase):
    _instance = None

    VALID_STAFF_EMAIL_RE = re.compile('^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$')
    VALID_STAFF_LOGIN_RE = re.compile('^[\w\-]{1,64}$')
    VALID_STAFF_WORK_PHONE_RE = re.compile('\d{4,5}')

    MAX_AGENT_ENTRIES_TO_CACHE = None

    ADMISSIBLE_STAFF_NAME_LOCALES = ('ru', 'en')

    ADMISSIBLE_STAFF_FILTERS = (
        'id', 'uid', 'email', 'username', 'external_username', 'work_phone',
        'user_id', 'yang_worker_id', 'first_name', 'last_name'
    )

    StaffSettingsEntry = collections.namedtuple(
        'StaffSettingsEntry',
        ('base_url', 'persons_api_url', 'api_token', 'name_locale', 'fields', 'request_limit')
    )

    StaffInfoEntry = collections.namedtuple(
        'StaffInfoEntry', ('username', 'print_name', 'work_phone', 'department_url')
    )

    def __init__(self, staff_settings, request_timeout=15, retries=3):
        super().__init__(request_timeout, retries)

        assert isinstance(staff_settings, self.StaffSettingsEntry)

        self._api_url = staff_settings.base_url + staff_settings.persons_api_url
        self._request_limit = staff_settings.request_limit
        self._fields_to_select = staff_settings.fields

        assert staff_settings.name_locale in self.ADMISSIBLE_STAFF_NAME_LOCALES
        self._name_locale = staff_settings.name_locale

        self._staff_info_cache = StaffInfoCache()

        self._setup_authorization_headers(staff_settings.api_token)

    @classmethod
    def make_default(cls):
        if cls._instance is None:
            cls._instance = cls.from_settings()
            cls._instance._staff_info_cache.invalidate_async()

        return cls._instance

    @classmethod
    def from_settings(cls):
        staff_settings = cls.StaffSettingsEntry(
            base_url=settings['base_url'],
            persons_api_url=settings['persons_api_url'],
            api_token=settings['token'],
            name_locale=settings['name_locale'],
            fields=','.join(settings['fields']),
            request_limit=settings['request_limit'],
        )

        request_timeout = settings['request_timeout']

        return cls(staff_settings, request_timeout)

    def format_staff_entry(self, entry):
        return StaffUserEntrySerializer(entry).data

    def format_user_entry(self, entry):
        return StaffUserEntrySerializer(entry).data

    def format_entry_generic(self, id_=None, login=None, print_name=''):
        return {
            'id': str(id_) if id_ is not None else id_,
            'user_id': str(id_) if id_ is not None else id_,
            'username': login,
            'print_name': print_name,
        }

    def try_extract_agent_work_phone(self, agent_trait):
        # ex.: Local/48490@ccm/n -> 48490, Agent/53165 -> 53165
        possible_agent_phone_ids = self.VALID_STAFF_WORK_PHONE_RE.findall(agent_trait)
        work_phone = possible_agent_phone_ids[0] if len(possible_agent_phone_ids) == 1 else None
        return work_phone

    def get_all_staff_entries(self, *, department_url_prefix=None):
        raw_entries = CallCenterStaffEntry.objects.all()

        if department_url_prefix is not None:
            raw_entries = raw_entries.filter(department_url__startswith=department_url_prefix)

        return raw_entries

    def invalidate_entries_cache(self):
        if self._staff_info_cache.active:
            self._staff_info_cache.stop_invalidation()

        self._staff_info_cache.invalidate_async()

    def get_agent_entry(self, **agent_filters):
        agent_filters = self._process_agent_filters(agent_filters)

        if len(agent_filters) >= 1:
            if len(agent_filters) == 1:
                filter_key, filter_value = next(iter(agent_filters.items()))
            else:
                filter_key, filter_value = tuple(agent_filters.keys()), tuple(agent_filters.values())

            agent_entry = self._staff_info_cache.get(filter_key, filter_value)

            if agent_entry is self._staff_info_cache.absent_object:
                agent_entry = self._get_db_agent_entry(**agent_filters)
                self._staff_info_cache.add(agent_entry, filter_key=filter_key, filter_value=filter_value)

        else:
            agent_entry = None

        return agent_entry

    def _get_db_agent_entry(self, **agent_filters):
        agent_entries = list(CallCenterStaffEntry.objects.filter(**agent_filters).order_by('id'))
        actual_agent_entry = self._pick_actual_agent_entry(agent_entries, agent_filters)
        return actual_agent_entry

    def get_agent_filters(self, agent_trait):
        agent_filter_kwargs = {}
        agent_trait = agent_trait.strip()

        if not agent_filter_kwargs:
            if self.VALID_STAFF_EMAIL_RE.match(agent_trait) is not None:
                agent_filter_kwargs = {'email': agent_trait}

        if not agent_filter_kwargs:
            possible_agent_phone_ids = self.VALID_STAFF_WORK_PHONE_RE.findall(agent_trait)

            if len(possible_agent_phone_ids) == 1:
                agent_filter_kwargs = {'work_phone': possible_agent_phone_ids[0]}

        if not agent_filter_kwargs:
            if self.VALID_STAFF_LOGIN_RE.match(agent_trait) is not None:
                agent_filter_kwargs = {'username': agent_trait}

        if not agent_filter_kwargs:
            LOGGER.error('unknown agent format cannot be processed: "{}"'.format(agent_trait))

        return agent_filter_kwargs

    def _process_agent_filters(self, agent_filters):
        agent_filters = {k: v for k, v in agent_filters.items() if v is not None}

        for filter_name in agent_filters:
            if filter_name not in self.ADMISSIBLE_STAFF_FILTERS:
                raise Exception('filter "{}" cannot be applied to obtain staff entry'.format(filter_name))

        return agent_filters

    def _pick_actual_agent_entry(self, agent_entries, agent_filters):
        active_agent_entries = [e for e in agent_entries if not (e.is_deleted or e.is_dismissed)]

        if active_agent_entries:
            if len(active_agent_entries) > 1:
                LOGGER.warning('there are more than 1 active agents "{}"'.format(agent_filters))
            entry = active_agent_entries[-1]
        elif agent_entries:
            entry = agent_entries[-1]  # all associated agents are dismissed; get the latest one
        else:
            LOGGER.warning('no information about agent "{}" is provided'.format(agent_filters))
            entry = None

        return entry

    def bind_external_username(self, external_username, internal_username, printable_name):
        entry = None

        if internal_username is not None and internal_username != 'null':
            entry = self.get_agent_entry(username=internal_username)

        if entry is None and printable_name is not None and printable_name != 'null':
            agent_name_parts = printable_name.split(' ')

            if len(agent_name_parts) >= 2:
                agent_last_name, agent_first_name = agent_name_parts[:2]
                entry = self.get_agent_entry(first_name=agent_first_name, last_name=agent_last_name)

        if entry is not None:
            with transaction.atomic(savepoint=False):
                db_entity = CallCenterStaffEntry.objects.select_for_update().get(id=entry.id)
                if db_entity.external_username != external_username:
                    db_entity.external_username = external_username
                    db_entity.save()

            self._staff_info_cache.add(entry)

        return entry

    def update_staff_entries(self, uid_collection, *, create_new_users=False):
        created_count = 0

        update_datetime = datetime_helper.utc_now()

        uid_request = ','.join(uid_collection)
        raw_staff_entries = self._get_staff_entries(uid=uid_request)

        with transaction.atomic(savepoint=False):
            for staff_entry in raw_staff_entries:
                has_created = self._update_staff_entry(staff_entry, update_datetime, create_new_users=create_new_users)
                created_count += int(has_created)

        total_count = len(raw_staff_entries)

        LOGGER.info('created {} new staff entries (total: {})'.format(created_count, total_count))

        return total_count, created_count

    def _get_staff_entries(self, *, uid=None, login=None, work_phone=None, work_email=None,
                           _sort_by_join_date=True):
        if uid is not None:
            selector_type, selector = 'uid', str(uid)
        elif login is not None:
            selector_type, selector = 'login', str(login)
        elif work_phone is not None:
            selector_type, selector = 'work_phone', str(work_phone)
        elif work_email is not None:
            selector_type, selector = 'work_email', str(work_email)
        else:
            raise Exception('no staff entry selector specified')

        fields_to_select = self._fields_to_select

        if _sort_by_join_date:
            fields_to_select += ',official.join_at'

        request_url = (
            '{api_url}{selector_type}={selector}&_fields=id,{fields_to_select}&_query=id%3e{min_id}&_sort=id&_nopage=1&_limit={limit}'
            .format(
                api_url=self._api_url,
                selector_type=selector_type,
                selector=selector,
                fields_to_select=fields_to_select,
                min_id='{min_id}',
                limit=self._request_limit
            )
        )

        all_pages_results = self._get_all_pages_results(request_url)

        if _sort_by_join_date:
            # join_at is a string representation of date, e.g. 1993-09-11 means Sep 11, 1993
            all_pages_results = sorted(all_pages_results, key=lambda e: e['official']['join_at'])

        return all_pages_results

    def _get_all_pages_results(self, request_url):
        all_pages_results = []

        current_page_url = request_url.format(min_id='-1')

        while current_page_url is not None:
            staff_entry_data = self._perform_request(current_page_url)
            current_page_results = staff_entry_data['result']

            if current_page_results:
                current_page_max_id = max(r['id'] for r in current_page_results)
                current_page_url = request_url.format(min_id=current_page_max_id)
                all_pages_results.extend(current_page_results)
            else:
                current_page_url = None

        return all_pages_results

    def _update_staff_entry(self, staff_entry, update_datetime, create_new_users):
        uid = staff_entry['uid']
        user = User.objects.filter(uid=uid).first()

        username = staff_entry['login']
        email = staff_entry['work_email']

        if user is None:
            if create_new_users:
                LOGGER.info('creating a new user for staff entry: uid={}, username={}, email={}'
                            .format(uid, username, email))
                User.objects.create(uid=uid, username=username, email=email)
            else:
                LOGGER.info('no user with uid {} can be paired with staff entry'.format(uid))
                return False

        mobile_phones = [e['number'] for e in staff_entry['phones'] if e['is_main']]
        mobile_phone = phone_number_helper.normalize_phone_number(mobile_phones[0]) if mobile_phones else None

        quit_at = staff_entry['official']['quit_at']
        if quit_at is not None:
            quit_at = datetime.datetime.strptime(quit_at, "%Y-%m-%d").date()

        extra_fields = {
            'uid': uid,
            'username': username,
            'first_name': (staff_entry['name']['first'][self._name_locale] or '').strip(),
            'last_name': (staff_entry['name']['last'][self._name_locale] or '').strip(),
            'middle_name': (staff_entry['name']['middle'] or '').strip(),
            'work_phone': staff_entry['work_phone'],
            'mobile_phone': mobile_phone,
            'email': email,
            'is_deleted': staff_entry['is_deleted'],
            'is_dismissed': staff_entry['official']['is_dismissed'],
            'department_url': staff_entry['department_group']['url'],
            'last_updated': update_datetime,
            'quit_at': quit_at,
        }

        has_created = False

        entry = CallCenterStaffEntry.objects.filter(user=user).first()

        if entry is None:
            entry = CallCenterStaffEntry.objects.create(user=user, **extra_fields)
            has_created = True
        else:
            for key, value in extra_fields.items():
                setattr(entry, key, value)
            entry.save()

        return has_created

    def get_staff_uids_from_specific_department(self, department_url):
        request_url = (
            '{api_url}_query='
            'id%3e{min_id}%20and%20'
            '(department_group.department.url=="{department_url}"%20or%20'
            'department_group.ancestors.department.url=="{department_url}")'
            '&_fields=id,uid&_sort=id&_nopage=1&_limit={limit}'
            .format(min_id='{min_id}', api_url=self._api_url, department_url=department_url, limit=self._request_limit)
        )

        all_pages_results = self._get_all_pages_results(request_url)
        staff_uids = [str(e['uid']) for e in all_pages_results]
        return staff_uids
