from django.db import models
from phonenumber_field.modelfields import PhoneNumberField

from cars.users.models import User


class CallCenterStaffEntry(models.Model):

    user = models.ForeignKey(
        User,
        on_delete=models.CASCADE,
        related_name='call_center_staff_entry',
    )

    uid = models.BigIntegerField(null=True)
    username = models.CharField(max_length=32)
    external_username = models.CharField(max_length=256, null=True)

    first_name = models.CharField(max_length=50)
    last_name = models.CharField(max_length=150)

    middle_name = models.CharField(max_length=64, blank=True)

    work_phone = models.IntegerField(null=True)  # work phone cannot be interpreted as ordinary phone
    mobile_phone = PhoneNumberField(null=True)
    email = models.EmailField()

    is_deleted = models.BooleanField(default=False)
    is_dismissed = models.BooleanField(default=False)

    department_url = models.CharField(max_length=150, null=True)
    yang_worker_id = models.CharField(max_length=40, null=True)

    last_updated = models.DateTimeField(null=True)  # last update from staff
    quit_at = models.DateField(null=True)

    class Meta:
        db_table = 'call_center_staff'

        indexes = [
            models.Index(
                fields=['work_phone'],
                name='cc_staff_work_phone_idx',
            ),
            models.Index(
                fields=['username'],
                name='cc_staff_username_idx',
            ),
            models.Index(
                fields=['email'],
                name='cc_staff_email_idx',
            ),
            models.Index(
                fields=['department_url'],
                name='cc_staff_department_url_idx',
            ),
        ]

    @property
    def print_name(self):
        return '{} {}'.format(self.first_name, self.last_name)

    @property
    def full_name(self):
        return ' '.join(field for field in (self.last_name, self.first_name, self.middle_name) if field)

    def __str__(self):
        return '<CallCenterStaffEntry: print_name={}, work_phone={}>'.format(self.print_name, self.work_phone)


class CallCenterStaffStatAccessEntry(models.Model):
    staff_entry = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.CASCADE,
    )

    outgoing_stat_granted_at = models.DateTimeField(
        null=True
    )

    outgoing_stat_refused_at = models.DateTimeField(
        null=True
    )

    class Meta:
        db_table = 'call_center_staff_stat_access'

    @property
    def access_granted(self):
        return self.outgoing_stat_granted_at is not None and self.outgoing_stat_refused_at is None

    def __str__(self):
        return ('<CallCenterStaffStatAccessEntry: staff_entry={}, granted_at={}>'
                .format(self.staff_entry_id, self.outgoing_stat_granted_at))
