import logging

from rest_framework.serializers import (
    CharField,
    DateTimeField,
    ModelSerializer,
    SerializerMethodField
)

from cars.django.serializers import BaseSerializer, TimestampField
from cars.orders.models import Order
from cars.users.models import User
from cars.callcenter.models import CallCenterStaffEntry
from .order import CallcenterOrderSerializer


LOGGER = logging.getLogger(__name__)


class UserPassportDataSerializer(BaseSerializer):

    first_name = CharField(default=None)
    last_name = CharField(default=None)
    patronymic_name = CharField(default=None, source='middle_name')
    birth_date = DateTimeField(default=None, format="%Y-%m-%d %H:%M:%S")
    birth_place = CharField(default=None)


class CallcenterUserSerializer(ModelSerializer):

    registered_at = TimestampField()
    passport_data = SerializerMethodField()
    order = SerializerMethodField()

    class Meta:
        model = User
        fields = [
            'username',
            'first_name',
            'last_name',
            'patronymic_name',
            'status',
            'registered_at',
            'phone',
            'is_phone_verified',
            'email',
            'passport_data',
            'order',
        ]

    def get_passport_data(self, obj):
        try:
            passport_raw_textdata = (
                self.context['datasync_client'].get_passport_unverified(
                    obj.uid, obj.passport_ds_revision
                ) or {}
            )
        except Exception:
            LOGGER.exception(
                'Unable to receive user passport data from verified DS, id=%s, uid=%s',
                str(obj.id),
                str(obj.uid),
            )
            passport_raw_textdata = {}

        return UserPassportDataSerializer(passport_raw_textdata).data

    def get_order(self, obj):
        current_order = (
            Order.objects
            .filter(
                user=obj
            )
            .order_by('-created_at')
            .first()
        )
        return CallcenterOrderSerializer(current_order).data


class RequestUserSerializer(ModelSerializer):
    print_name = SerializerMethodField(initial='')

    class Meta:
        model = User
        fields = [
            'id',
            'uid',
            'username',
            'print_name',
            'status',
            'phone',
            'is_phone_verified',
        ]

    def get_initial(self):
        initial = super().get_initial()

        for readonly_field in ('print_name', ):
            initial[readonly_field] = self.fields[readonly_field].initial

        return initial

    def get_print_name(self, obj):
        return obj.get_print_name()


class StaffEntrySerializer(ModelSerializer):
    class Meta:
        model = CallCenterStaffEntry
        fields = [
            'username',
            'print_name',
            'work_phone',
            'department_url',
        ]


class StaffUserEntrySerializer(BaseSerializer):
    id = SerializerMethodField(initial=None)
    user_id = SerializerMethodField(initial=None)
    username = SerializerMethodField(initial=None)
    print_name = SerializerMethodField(initial='')

    def get_initial(self):
        initial = super().get_initial()

        for readonly_field in ('id', 'user_id', 'username', 'print_name', ):
            initial[readonly_field] = self.fields[readonly_field].initial

        return initial

    def get_id(self, obj):
        return self.get_user_id(obj)

    def get_user_id(self, obj):
        return getattr(obj, 'user_id', getattr(obj, 'id'))

    def get_username(self, obj):
        return obj.username

    def get_print_name(self, obj):
        if hasattr(obj, 'get_print_name'):
            return obj.get_print_name()

        return getattr(obj, 'print_name', '')
