from rest_framework.response import Response
from rest_framework.status import HTTP_401_UNAUTHORIZED, HTTP_403_FORBIDDEN

from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.users.core.datasync import DataSyncDocumentsClient
from ..core.authentication import CallcenterAuthentication


class CallcenterAPIView(CarsharingAPIView):

    datasync = DataSyncDocumentsClient.from_settings()
    permission_classes = [IsAuthenticated]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.auth_verifier = CallcenterAuthentication.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['datasync_client'] = self.datasync
        return context

    def get(self, request, *args, **kwargs):
        error_response = self._get_error_response(request, kwargs.get('call_id'))
        if error_response is not None:
            return error_response

        return super().get(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        error_response = self._get_error_response(request, kwargs.get('call_id'))
        if error_response is not None:
            return error_response

        return super().post(request, *args, **kwargs)

    def _get_error_response(self, request, call_id):
        if request.user.is_anonymous:
            return Response(
                data={
                    'reason': 'Anonymous access is not allowed.',
                },
                status=HTTP_401_UNAUTHORIZED,
                content_type='application/json',
            )

        is_allowed, error_reason = self.auth_verifier.is_allowed(request.user.uid, call_id)
        if not is_allowed:
            return Response(
                data={
                    'reason': error_reason,
                },
                status=HTTP_403_FORBIDDEN,
                content_type='application/json',
            )

        return None
