from abc import abstractmethod

from rest_framework.response import Response

import cars.settings
from cars.core.util import import_class
from cars.orders.models import OrderItem
from ..models import CallcenterUserLink
from .base import CallcenterAPIView


class CarCommandView(CallcenterAPIView):

    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def do_post(self, request, call_id):  # pylint: disable=unused-argument
        user = CallcenterUserLink.objects.filter(id=call_id).first().user
        latest_order_item = (
            OrderItem.objects
            .filter(order__user=user)
            .order_by('-started_at')
            .first()
        )
        car = latest_order_item.get_impl().car

        self.do_command(car)
        return Response()

    @abstractmethod
    def do_command(self, car):
        raise NotImplementedError


class OpenCarCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.open(car.imei)

class CloseCarCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.close(car.imei)

class UnlockHoodCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.unlock_hood(car.imei)

class LockHoodCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.lock_hood(car.imei)

class StartLeaseCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.start_lease(car.imei)

class EndLeaseCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.end_lease(car.imei)
