from abc import abstractmethod

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.carsharing.models import Car
from cars.eka.core.processor import FuelCardProcessor
from cars.eka.models import FuelCardActivation
from cars.orders.models import Order
from ..models import CallcenterUserLink
from ..serializers import CallcenterCarSerializer, UpdateFuelCardNumberViewArgumentsSerializer
from .base import CallcenterAPIView


class GenericFuelCardView(CallcenterAPIView):

    def do_post(self, request, call_id):  # pylint: disable=unused-argument
        user = CallcenterUserLink.objects.filter(id=call_id).first().user
        order = (
            Order.objects
            .filter(user=user)
            .order_by('-created_at')
            .first()
        )
        if order is not None:
            car = (
                order.items
                .order_by('-started_at')
                .first()
                .get_impl()
                .car
            )
        else:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        processor = FuelCardProcessor.from_order(
            order=order,
            operator=request.user,
            source=FuelCardActivation.Source.EXTERNAL_CC.value,
        )

        self._do_fuel_card_action(processor)

        updated_car = Car.objects.get(id=car.id)

        return Response(CallcenterCarSerializer(updated_car).data)

    @abstractmethod
    def _do_fuel_card_action(self, processor):
        raise NotImplementedError


class UpdateFuelCardNumberView(GenericFuelCardView):

    arguments_serializer_class = UpdateFuelCardNumberViewArgumentsSerializer

    def _do_fuel_card_action(self, processor):
        processor.change_card_number(self.request.arguments['number'])


class FuelCardActivateView(GenericFuelCardView):

    def _do_fuel_card_action(self, processor):
        processor.activate_fuel_card()


class FuelCardBlockView(GenericFuelCardView):

    def _do_fuel_card_action(self, processor):
        processor.block_fuel_card()
