from django.shortcuts import get_object_or_404
from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from rest_framework.status import HTTP_201_CREATED

import cars.settings
from ..core import (
    ACLVerifier,
    CallcenterAuthentication,
    CallcenterLinkManager,
)
from ..models import CallcenterUserLink
from ..serializers import CallcenterRequestArgumentsSerializer
from .base import CallcenterAPIView


class CallcenterRequestView(CallcenterAPIView):

    manager = CallcenterLinkManager()

    arguments_serializer_class = CallcenterRequestArgumentsSerializer

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.auth_verifier = CallcenterAuthentication(
            acl_verifier=ACLVerifier(
                acl=[
                    cars.settings.CALLCENTER['call_registration_server_uid'],
                ]
            ),
            call_id_verifier=None,
        )

    def do_post(self, request):
        phone = request.arguments.get('phone')
        operator_uid = request.arguments.get('operator_uid')

        request, error_msg = self.manager.create_request_link(operator_uid)

        if request is None:
            return Response(
                {
                    'status': 'error',
                    'code': error_msg,
                }
            )

        try:
            self.manager.register_phone_number(request, phone)
            result_url = cars.settings.CALLCENTER['link_patterns']['user_card'].format(
                str(request.id)
            )
        except NotFound:
            result_url = cars.settings.CALLCENTER['link_patterns']['enter_phone'].format(
                str(request.id)
            )
        request.save()

        return Response(
            {
                'url': result_url,
                'status': 'ok',
            },
            status=HTTP_201_CREATED
        )


class CallcenterRegisterPhoneView(CallcenterAPIView):

    manager = CallcenterLinkManager()

    arguments_serializer_class = CallcenterRequestArgumentsSerializer

    def do_post(self, request, call_id):
        phone = request.arguments.get('phone')
        self.manager.register_phone_number(
            request=get_object_or_404(CallcenterUserLink, id=call_id),
            phone=phone,
        )
        return Response()
