from django.shortcuts import get_object_or_404
from rest_framework.exceptions import PermissionDenied

import cars.settings
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.orders.models import Order
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.models import User


class CallcenterInternalAPIView(CarsharingAPIView):

    acl = cars.settings.CALLCENTER_INTERNAL['acl']
    datasync = DataSyncDocumentsClient.from_settings()

    permission_classes = [IsAuthenticated]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.user = None
        self.car = None
        self.order = None

        self.saas_client = SaasDriveAdminClient.from_settings()

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['datasync_client'] = self.datasync
        return context

    def initial(self, request, *args, **kwargs):
        if not self._check_access(request.user):
            raise PermissionDenied

        self._load_connected_objects(kwargs['phone_number'])

        return super().initial(request, *args, **kwargs)

    def _check_access(self, user):
        if self._check_acl(user):
            return True
        has_required_role = self.saas_client.check_user_role(cars.settings.CALLCENTER_INTERNAL['role'], user_id=user.id)
        return has_required_role

    def _check_acl(self, user):
        if self.acl is None:
            return True
        return user.is_authenticated and user.uid in self.acl

    def _load_connected_objects(self, phone_number):
        self.user = get_object_or_404(
            User,
            phone='+{}'.format(phone_number),
            is_phone_verified=True,
        )
        self.order = (
            Order.objects
            .filter(user=self.user)
            .order_by('-created_at')
            .first()
        )
        if self.order is not None:
            self.car = (
                self.order.items
                .order_by('-started_at')
                .first()
                .get_impl()
                .car
            )
        else:
            self.car = None
